 #ifndef DEVPCIEDEFS_H
#define DEVPCIEDEFS_H
/*
// ==========================================================================
//
// = LIBRARY
//     shared/dev
// 
// = FILENAME
//    devpciedefs.h
//
// = COPYRIGHT
//    (C) Copyright Agilent Technologies 2003
//
// ==========================================================================
*/

#include "devmidl.h"

// -----------------------------------------------------------------------
// <A NAME="EAgtPatternType">EAgtPatternType</A>

/// \addtogroup Control
///
/// @{

/// \brief Information about the fpga
typedef DEV_MIDL_ENUM enum EPCIEFPGAInfo
{
  PCIE_FPGA_VERSION_NUM,///< This is the FPGA version number. The version number of the FPGAs is formatted as: Bits7:0 contain the 'Minor Version Number' and Bits23:8 contain the 'Major Version Number'.
  PCIE_FPGA_TYPE ///< The FPGA type. Please see \ref EPCIEFPGAType
} EPCIEFPGAInfo;

/// \brief FPGA Types
///
/// The number in parentheses will be returned when using \ref FPGAInfoGet with this enum.
typedef DEV_MIDL_ENUM enum EPCIEFPGAType
{
  PCIE_FPGA_POLLUX = 1, ///< (1) This is a Pollux (Exerciser for PCI Express Transmit FPGA)
  PCIE_FPGA_CASTOR = 2, ///< (2) This is a Castor (Exerciser for PCI Express Receive FPGA) 
  PCIE_FPGA_HELENA2 = 4,///< (4) This is a Helena2 (Analyzer for PCI Express FPGA)
  PCIE_FPGA_HELENA1 = 5, ///< (5) This is a Helena1 (Analyzer for PCI Express FPGA)
  PCIE_FPGA_POLLUX_GHI = 8, ///< (8) This is a GHI Pollux (Exerciser for PCI Express GHI Transmit FPGA)
  PCIE_FPGA_CASTOR_GHI = 9, ///< (9) This is a GHI Castor (Exerciser for PCI Express GHI Receive FPGA) 
  PCIE_FPGA_HELENA1_GHI = 10, ///< (10) This is a Helena1 GHI (Analyzer for PCI Express GHI FPGA)
  PCIE_FPGA_HELENA2_GHI = 11, ///< (11) This is a Helena2 GHI (Analyzer for PCI Express GHI FPGA)
  PCIE_FPGA_HAWK_FRONT = 12,   ///< (12) This is a Hawk Front (Exerciser for PCI Express Gen2) 
  PCIE_FPGA_CONDOR = 14,      ///< (14) This is the FPGA for the N5309 board
  PCIE_FPGA_TRISUL = 15      ///< (15) This is the FPGA for the U4305A Gen3 Exercser board
} EPCIEFPGAType;

/// \brief Probe board modes
///
/// The numbers indicated in parentheses are the values needed to read/write the ProbeBoardMode.
typedef DEV_MIDL_ENUM enum EPCIEProbeBoardMode
{
  PCIE_PROBEBOARDMODEV_EXE_TO_UPSTREAM = 1,
  ///< (1) Probeboard 'Exerciser To Upstream' mode. The exerciser is
  ///communicating via the system connector ("card-edge") of the probeboard.
  PCIE_PROBEBOARDMODEV_EXE_TO_UPSTREAM_LOOPBACK = 2,
  ///< (2) Probeboard 'Exerciser To Upstream Loopback' mode. The exerciser is
  ///acting as an upstream port, which is looped back to itself. Can be used
  ///for demonstration purposes or for testing test programs.
  PCIE_PROBEBOARDMODEV_EXE_TO_DOWNSTREAM = 3,
  ///< (3) Probeboard 'Exerciser To Downstream' mode. The exerciser is
  ///communicating via the top (DUT) connector of the probeboard.
  PCIE_PROBEBOARDMODEV_EXE_TO_DOWNSTREAM_LOOPBACK = 4,
  ///< (4) Probeboard 'Exerciser To Downstream Loopback' mode. The exerciser
  ///is acting as a downstream port, which is looped back to itself. Can be
  ///used for demonstration purposes, or for testing test programs.
  PCIE_PROBEBOARDMODEV_ANALYZER = 5,
  ///< (5) Probeboard 'Analyzer' mode.
  PCIE_PROBEBOARDMODEV_ANALYZER_LOOPBACK_TO_DUT = 6,
  ///< (6) Probeboard 'Analyzer Loopback to DUT' mode. The top (DUT) connector
  ///of the probe board is in loopback. This allows testing an add-in card in
  ///loopback mode, while having an Analyzer to monitor the traffic. Note that
  ///both directions of the Analyzer will show the same data.
  PCIE_PROBEBOARDMODEV_ANALYZER_LOOPBACK_TO_SYS = 7,
  ///< (7) Probeboard 'Analyzer Loopback to SYS' mode. The bottom (SYS)
  ///connector of the probe board is in loop-back. This allows testing an
  ///add-in card in loopback mode, while having an Analyzer to monitor the
  ///traffic. Note that both directions of the Analyzer will show the same
  ///data.
  PCIE_PROBEBOARDMODEV_DISCONNECT = 8,
  ///< (8) Probeboard 'Disconnect' mode. All outputs are disabled.
  PCIE_PROBEBOARDMODEV_EXE_TO_AMC=9,
  ///< (ATCA probe only) The exerciser is communicating via the AMC connector
  PCIE_PROBEBOARDMODEV_ANALYZER_AMC=10
  ///< (ATCA probe only) The module in the AMC connector is connected to the
  ///< system/chassis connector. The protocol Analyzer monitors traffic in both directions.

} EPCIEProbeBoardMode;

/// \brief Probe board actions
///
typedef DEV_MIDL_ENUM enum EPCIEProbeBoardExecute
{
  PCIE_PROBEBOARDEXECUTE_PCIE_RESET ///< Executes a PCIE Reset on the probe board. 
} EPCIEProbeBoardExecute;

/// \brief Probe board settings
///
/// The numbers indicated in parentheses are the values needed to read/write the ProbeBoard.
typedef DEV_MIDL_ENUM enum EPCIEProbeBoard
{
  PCIE_PROBEBOARD_USED_LINKWIDTH,
  ///< Number of lanes enabled on the probe board. Values are:
  ///(0x1=x1 link; 0x2=x2 link; 0x4=x4 link; 0x8=x8 link).
  /// \note  Calling IAgtPCIEExerciser::DllPhySet with parameter
  /// PCIE_DLLPHY_LINKMODE_CAPABLE will automatically set this property to the
  /// highest needed value.
  /// \note Unlike PCIE_DLLPHY_LINKMODE_CAPABLE, this is not a bit field. Only
  /// distinct values are allowed. 
  PCIE_PROBEBOARD_PHYS_LINKWIDTH,
  ///< Number of lanes physically available on the probe board. Values are:
  ///(0x1=x1 link ;0x2=x2 link ;0x4=x4 link; 0x8=x8 link).
  PCIE_PROBEBOARD_REVISION ,
  ///< Revisionnumber 
  PCIE_PROBEBOARD_CLOCKSETTING ,
  // Clock setting (selftest). Internal use only.
  PCIE_PROBEBOARD_PRESENT_PINS ,
  // Bitfield of probeboard pins (selftest). Internal use only.
  PCIE_PROBEBOARD_DISPLAY ,
  // Text for probeboard display (selftest). Internal use only.
  PCIE_PROBEBOARD_CLOCK_INPUT_SELECTOR,
  ///< This selects the source for the input clock of the probe board. The clock can either be 
  ///taped from the system connector or the  external-in (SMA) connector.
  PCIE_PROBEBOARD_CLOCK_OUTPUT_SELECTOR,
  ///< This selects the source for the clock provided at the DUT connector of the probe board. 
  /// The clock can either be provided by the IO Module or the Clock Input Selector 
  /// (see \ref PCIE_PROBEBOARD_CLOCK_INPUT_SELECTOR) on the probe board.
  PCIE_PROBEBOARD_FAST_SYNC_MODE,
  ///< Enables (1) or disables (0) the fast sync mode on passive and midbus probe boards.
  ///  Default is 0 except when GHI images are used.
  ///  Set this to 1 when using data rates below 2GBit/s.  
  PCIE_PROBEBOARD_MODE ,
  ///< Use this property to modify ProbeBoard mode. For details, please refer to \ref EPCIEProbeBoardMode.
  PCIE_PROBEBOARD_ID ,
  // Get ProbeBoard id 
  PCIE_PROBEBOARD_CHANNEL
  ///< (ATCA Probe only) the channel which is monitored by the analyzer or used by the exerciser to communicate:
  /// Default: 1, legal values 1-4, 14, 15. Only valid for probe modes "_EXE_TO_UPSTREAM", "_EXE_TO_DOWNSTREAM", 
  /// and "_ANALYZER". For exerciser modes, the exerciser will be connected to the respective channel,
  /// and all other channels will be disconnected. 
  /// For analyzer modes, all connections remain, but the channel the analyzer listens to is changed.

} EPCIEProbeBoard;

/// \brief Probe board clock input selector values
///
/// The numbers indicated in parentheses are the values needed to read/write the ProbeBoard.
typedef DEV_MIDL_ENUM enum EPCIEProbeBoardClockValInput
{
  PCIE_PROBEBOARD_CLOCKV_INPUT_SYSTEM_CONNECTOR = 0, ///< (0) The clock is taped from the system connector.
  PCIE_PROBEBOARD_CLOCKV_INPUT_EXT_IN_CONNECTOR = 1  ///< (1) The clock is taped from the external-in (SMA) connector.
} EPCIEProbeBoardClockValInput;

/// \brief Probe board clock output selector values
///
/// The numbers indicated in parentheses are the values needed to read/write the ProbeBoard.
typedef DEV_MIDL_ENUM enum EPCIEProbeBoardClockValOutput
{
  PCIE_PROBEBOARD_CLOCKV_OUTPUT_IOMODULE = 0, ///< (0) The clock provided at the DUT connector of the probe board comes from the IO Module.
  PCIE_PROBEBOARD_CLOCKV_OUTPUT_CLOCK_INPUT_SELECTOR = 1///< (1) The clock provided at the DUT connector comes from the clock input selector (see \ref PCIE_PROBEBOARD_CLOCK_INPUT_SELECTOR)
} EPCIEProbeBoardClockValOutput;

/// \brief IO Module properties
///
/// Please be aware that the Exerciser cannot be programmed as long as there is no valid clock available. 
/// If the Probe Board is plugged into a DUT and you need to use clock of this DUT, the following steps
/// are recommended: 
/// - Program the exerciser
/// - Switch to external clock
/// - Call 'LinkUp'
/// - Power the DUT
typedef DEV_MIDL_ENUM enum EPCIEModule
{
  PCIE_MODULE_EXTERNAL_CLOCK_ENABLE ///< Enables the external clock. 0 -- use
                                    ///internal clock, 1 -- use external
                                    ///clock. Default: 0
} EPCIEModule;

/// \brief IO Module info
///
/// Returns information for the IO Module
typedef DEV_MIDL_ENUM enum EPCIEModuleInfo
{
  PCIE_MODULEINFO_MAXIMUM_LINKWIDTH_SUPPORTED /// < This is the maximum linkwidth that can be established. This takes into account: Session Type (Analyzer/Exerciser) and software packages available.
} EPCIEModuleInfo;


/// @} end of group Control in this file


/// \addtogroup Exerciser
///
/// @{

// this is provided in the ESW only
#ifndef COMPILING_DOC
typedef DEV_MIDL_ENUM enum EPCIEHostControllerVersion
{
  PCIE_HOSTCONTROLLERVERSION_MAJOR = 0,  
  PCIE_HOSTCONTROLLERVERSION_MINOR = 1,
  PCIE_HOSTCONTROLLERVERSION_RELEASE = 2,
  PCIE_HOSTCONTROLLERVERSION_BUILD = 3,
} EPCIEHostControllerVersion;
#endif

#ifndef COMPILING_DOC
/// \brief Capture Memory
///
/// Available memories for PTC Capture. Used by CaptureMemoryRead().
typedef DEV_MIDL_ENUM enum EPCIECaptureMemory
{
  PCIE_CAPTURE_MEMORY_TLPDATA,
  ///< This memory contains TLP header data, i.e. the first 4 DWs of each captured TLP.
  /// Valid after capturing has been stopped.
  PCIE_CAPTURE_MEMORY_DLLPDATA,
  ///< This memory contains captured DLLPs (each one DW).
  /// Valid after capturing has been stopped.
  PCIE_CAPTURE_MEMORY_TLPSTAMP,
  ///< This memory contains TLP timestamps; one DW for each captured TLP.
  /// Valid after capturing has been stopped.
  /// Caution: These TLP timestamps are not correlated to the DLLP timestamps (see PCIE_CAPTURE_MEMORY_DLLPSTAMP).
  PCIE_CAPTURE_MEMORY_DLLPSTAMP,
  ///< This memory contains DLLP timestamps; one DW for each captured DLLP.
  /// Valid after capturing has been stopped.
  /// Caution: These DLLP timestamps are not correlated to the TLP timestamps (see PCIE_CAPTURE_MEMORY_TLPSTAMP).
  PCIE_CAPTURE_MEMORY_TLPQUAL, // Internal use only
  PCIE_CAPTURE_MEMORY_DLLPQUAL // Internal use only
} EPCIECaptureMemory;

/// \brief Capture Status
///
/// Status of Capture memory
typedef DEV_MIDL_ENUM enum EPCIECaptureStatus
{
  PCIE_CAPTURE_STATUS_RESERVED0,
  // Internal use only. Contents of CaptStatus register in FPGA.
  PCIE_CAPTURE_STATUS_STATUS,
  ///< Current status of capture memory.
  ///  0=stopped:   Initial state where patterns can be programmed.
  ///  1=waiting:   Capturing has been started. Not yet triggered.
  ///  2=triggered: At least one pattern matcher has triggered; no packet stored yet.
  ///  3=storing:   At least one TLP or DLLP has been stored.
  ///  4=finished:  Capturing has been stopped, either manually (by calling CaptureStop()) or because TLP or DLLP memory is already full.
  //   0xffffffff:  Error: Invalid status, should never happen 
  PCIE_CAPTURE_STATUS_NUMTLP,
  ///< Number of TLPs captured. Needed for CaptureMemoryRead(). Read this value after capturing has stopped.
  PCIE_CAPTURE_STATUS_NUMDLLP,
  ///< Number of DLLPs captured. Needed for CaptureMemoryRead(). Read this value after capturing has stopped.
  PCIE_CAPTURE_STATUS_ERRORTLPLENGTH
  ///< Whether a TLP length error has occurred during capturing
} EPCIECaptureStatus;

/// \brief Capture Status
///
/// Status calues for property PCIE_CAPTURE_STATUS_STATUS
typedef DEV_MIDL_ENUM enum EPCIECaptureStatusVal
{
  PCIE_CAPTURE_STATUS_STATUS_STOPPED=0,
  ///< Initial state where patterns can be programmed.
  PCIE_CAPTURE_STATUS_STATUS_WAITING=1,
  ///< Capturing has been started. Not yet triggered.
  PCIE_CAPTURE_STATUS_STATUS_TRIGGERED=2,
  ///< At least one pattern matcher has triggered. No packet stored yet.
  PCIE_CAPTURE_STATUS_STATUS_STORING=3,
  ///<  At least one TLP or DLLP has been stored.
  PCIE_CAPTURE_STATUS_STATUS_FINISHED=4,
  ///< Capturing has been stopped, either manually (by calling CaptureStop()) or because TLP- or DLLP memory is already full.
  PCIE_CAPTURE_STATUS_STATUS_ERROR=0xffffffff
  //  Invalid status, should never happen 
} EPCIECaptureStatusVal;

/// \brief Capture Pattern properties
///
/// Properties to configure the behavior of a pattern matcher
typedef DEV_MIDL_ENUM enum EPCIECapturePattern
{
  PCIE_CAPTURE_PATTERN_ENABLED,
  ///< 0: Pattern matcher is disabled. This means it will never trigger or store the captured data. This is the default setting. 
  ///  1: Pattern matcher is enabled and will trigger the capture memory in case of a match. Matching packet will not be stored.
  ///  2: Pattern matcher is enabled. Matching packets will be stored, if trigger has already occurred.
  ///  3: Pattern matcher is enabled. Matching packets will trigger the capture memory (if not already occured) and be stored.
  PCIE_CAPTURE_PATTERN_VALUE,
  ///< Value of pattern matcher (32bit).
  /// For pattern matchers 0-3, this is the first DW of a TLP and for pattern matchers 4-7, this is the first DW of a DLLP.
  /// Caution: Only bits for which the corresponding mask (PCIE_CAPTURE_PATTERN_MASK) is set to '1' are taken into account.
  PCIE_CAPTURE_PATTERN_MASK
  ///< Mask of pattern matcher (32bit). 
  /// Determines which bits of the value (PCIE_CAPTURE_PATTERN_VALUE) are taken into account for triggering/capturing.
  /// A mask bit with value '0' means that the corresponding bit of the TLP/DLLP is not matched with the bit from PCIE_CAPTURE_PATTERN_VALUE. 
  /// A mask bit with value '1' means that the corresponding bit of the TLP/DLLP must match the bit from PCIE_CAPTURE_PATTERN_VALUE. 
} EPCIECapturePattern;

#endif    // #ifndef COMPILING_DOC


/// \brief Decoder properties
///
/// Properties of the Base Address Decoders. 
typedef DEV_MIDL_ENUM enum EPCIEDecProp
{
  PCIE_DECP_RESOURCE    = 0,
  ///< Resource for data (memory or data generator), possible property values:
  ///  \ref EPCIEDecValResource
  PCIE_DECP_DEC,
  ///< Decoder type (enabled/disabled), possible property values: \ref
  ///  EPCIEDecValDec
  PCIE_DECP_LOC,
  ///< Location (IO, Memory 32-bit or 64-bit), possible property values: \ref EPCIEDecValLoc. This enum entry is not valid for the Expansion ROM decoder and the subtractive decoder (see \ref EPCIEDec).
  PCIE_DECP_PREFETCHABLE,
  ///< Data is prefetchable (allowed for memory decoders only)
  PCIE_DECP_QUEUE,
  ///< Completion queue (set if completion is needed), possible property values: \ref EPCIECompQueue
  PCIE_DECP_SIZE,
  ///< Size of decoder. Value to be set is n, the decoder size is 2^n. The
  /// value of n is dependent on the decoder type (MEM32: 3 < n < 32, IO: 1 < n
  /// < 32, MEM64: 3 < n < 64, ExpRom: 10 < n < 25)
  PCIE_DECP_BASEADDR_LO,
  ///< Base address of decoder. Lower dword of 64-bit address. This enum entry is not valid for the Subtractive decoder (see \ref EPCIEDec).
  PCIE_DECP_BASEADDR_HI,
  ///< Base Address of decoder. Upper dword of 64-bit address. Only to be used
  ///  for 64bit decoder. Address will show in the next BAR. Cannot be used on
  ///  decoder 5. This enum entry is not valid for the Expansion ROM decoder and the subtractive decoder (see \ref EPCIEDec).
  PCIE_DECP_INTADDR,
  ///< Internal address. This is the memory address where the data is
  ///stored/read in the exerciser memory.
  PCIE_DECP_INTSIZE,
  ///< Size of the internal memory used for this decoder. Value to be set is n, the size is 2^n bytes. 
  ///The  range of n is: 4 < n < 19.
} EPCIEDecProp;

/// \brief Available Completer queues
///
/// Possible values of decoder property PCIE_DECP_QUEUE (see \ref EPCIEDecProp).
typedef DEV_MIDL_ENUM enum EPCIECompQueue
{
  PCIE_COMPQUEUE_0 = 0,         ///< Completer queue 0
  PCIE_COMPQUEUE_1,             ///< Completer queue 1
  PCIE_COMPQUEUE_2,             ///< Completer queue 2 (*New in Gen3 Exerciser*)
  PCIE_COMPQUEUE_3,             ///< Completer queue 3 (*New in Gen3 Exerciser*)
  PCIE_COMPQUEUE_4,             ///< Completer queue 4 (*New in Gen3 Exerciser*)
  PCIE_COMPQUEUE_END
} EPCIECompQueue;

/// \brief Decoder settings
///
/// Possible values of decoder property PCIE_DECP_DEC (see \ref EPCIEDecProp).
///
/// Using Tcl, you will not be able to use the enumeration. Please use the
/// number indicated in parentheses instead.
///
/// \note Please be aware that for a decoder to be active, the following bits
/// in the Config Space must be set to 1 as well (use \ref ConfigSpaceWrite):
/// For a Memory Decoder, the 'Memory Access Enable Bit'. For an IO Decoder, the
/// 'IO Access Enable Bit'.  For the Expansion Rom Decoder, the 'Memory Enable
/// Bit' and 'Expansion Rom Enable Bit'.
typedef DEV_MIDL_ENUM enum EPCIEDecValDec
{
  PCIE_DECV_DEC_DISABLE = 0,   ///< (0) Disable the decoder.
  PCIE_DECV_DEC_ENABLE  = 1,   ///< (1) Enable the decoder.
} EPCIEDecValDec;

/// \brief Decoder resources
///
/// Possible values of decoder property PCIE_DECP_RESOURCE (see \ref EPCIEDecProp).
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIEDecValResource
{
  PCIE_DECV_RESOURCE_MEM = 1,           ///< (1) Incoming data is written to data memory, outgoing data is read from data memory.
  PCIE_DECV_RESOURCE_GENERATOR = 2,     ///< (2) Incoming data is compared to generated data, outgoing data is generated.
  PCIE_DECV_RESOURCE_DISCARD = 3,       ///< (3) Incoming data is discarded, outgoing data is read from data memory. (Values are not guaranteed to be valid).
  PCIE_DECV_RESOURCE_MEM_COMPARE = 4,    ///< (4) Incoming data is compared to memory and discarded, outgoing data is read from data memory.
} EPCIEDecValResource;

/// \brief Block resources
///
/// Possible values of decoder property PCIE_BLOCK_RESOURCE (see \ref EPCIEBlock).
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIEBlockValResource
{
  PCIE_BLOCKV_RESOURCE_MEM = 1,           ///< (1) Incoming data is written to data memory and outgoing data is read from data memory.
  PCIE_BLOCKV_RESOURCE_GENERATOR = 2,     ///< (2) Incoming data is compared to generated data and outgoing data is generated.
  PCIE_BLOCKV_RESOURCE_DISCARD = 3       
} EPCIEBlockValResource; 


/// Possible values of block property PCIE_BLOCK_WAITREPEAT (see \ref EPCIEBlock). Cannot be ORed.
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIEBlockValWaitRepeat
{
  // Please do not change the enum values without checking the HW registers !
  PCIE_BLOCKV_WAITREPEAT_REPEATPATTERN  = 0, ///< (0) Default. The blockline is repeated as long as all enabled pattern terms (see PCIE_BLOCK_PATTERN_TERM in \ref EPCIEBlock) are false. If all patterns are disabled (PCIE_BLOCK_PATTERN_TERM=0), the blockline will not be repeated.
  PCIE_BLOCKV_WAITREPEAT_WAITPATTERN    = 4, ///< (4) The execution of the blockline is delayed as long as all enabled pattern terms (see PCIE_BLOCK_PATTERN_TERM in \ref EPCIEBlock) are false. If all patterns are disabled (PCIE_BLOCK_PATTERN_TERM=0), the blockline will not be delayed.
  PCIE_BLOCKV_WAITREPEAT_WAITCOMPLETION = 5, ///< (5) The execution of the blockline is delayed until all outstanding completions have been finished. If PCIE_BLOCK_PATTERN_TERM>0, it will additionally wait as long as all enabled pattern terms are false.
  PCIE_BLOCKV_WAITREPEAT_WAITTRIGGER    = 6  ///< (6) The execution of the blockline is delayed until an external trigger event occurs. This can be achieved by connecting an external instrument to the trigger-in port. See description of trigger connector. If PCIE_BLOCK_PATTERN_TERM>0, it will additionally wait as long as all enabled pattern terms are false.
} EPCIEBlockValWaitRepeat;

/// \brief Decoder Location
///
/// Possible values of decoder property PCIE_DECP_LOC (see \ref EPCIEDecProp).
///
/// The location that this decoder should map to:
///  -# Memory
///      - 32-bit
///      - 64-bit
///  -# I/O
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIEDecValLoc
{
  PCIE_DECV_LOC_MEM32 = 0,    ///< (0) 32-bit memory decoder
  PCIE_DECV_LOC_MEM64 = 1,    ///< (1) 64-bit memory decoder, lower 32 bits
  PCIE_DECV_LOC_MEM64_HI = 2, ///< (2) 64-bit memory decoder, upper 32 bits
  PCIE_DECV_LOC_IO = 3        ///< (3) I/O Decoder
} EPCIEDecValLoc;

/// \brief BAR Number to map to
///
/// This is the BAR to which the actual decoder maps to.
typedef DEV_MIDL_ENUM enum EPCIEDec
{
  PCIE_DEC_BAR0 = 0,          ///< BAR 0
  PCIE_DEC_BAR1 = 1,          ///< BAR 1
  PCIE_DEC_BAR2 = 2,          ///< BAR 2
  PCIE_DEC_BAR3 = 3,          ///< BAR 3
  PCIE_DEC_BAR4 = 4,          ///< BAR 4
  PCIE_DEC_BAR5 = 5,          ///< BAR 5
  PCIE_DEC_EXP  = 6,          ///< Expansion ROM decoder
  PCIE_DEC_SUB  = 7,          ///< Subtractive decoder. This decoder is not available in Gen3 Exerciser.
} EPCIEDec;

/// \brief "Send Immediate" packet status values
///
/// Status values from packet send
typedef DEV_MIDL_ENUM enum EPCIESiStatus
{
  PCIE_SISTATUS_COMP_AVAILABLE, ///< The completion for a Send Immediate
                                ///request is available
  PCIE_SISTATUS_SENT         ///< 1 The Send Immediate request has been sent
                             ///< 0 The Send Immediate request has not been sent
} EPCIESiStatus;

/// \brief "Send Immediate" packet properties.
///
/// Properties in a "Send Immediate" packet.
/// For properties that are represented in the TLP, you will find additional information about the location in the header.
/// For example: "bits[3:2] of byte 2" means: This value will appear in bits 2 und 3 of byte 2 in the TLP header.
typedef DEV_MIDL_ENUM enum EPCIESi
{
    PCIE_SI_LEN = 0,  ///< Length of data payload in DWORDs, maximum is 1024 (bits [1:0] of byte 2, bits [7:0] of byte 3). Please note: 0 indicates 1024 DWORDs, so the valid range of values for this field is 0-1023. Please adapt 'PCIE_SI_LASTDWBE' accordingly when modifying this value. (default 0x1) 
    PCIE_SI_RSVD023,  ///< Reserved (bits [3:2] of byte 2) (default 0x0)
    PCIE_SI_ATTR,     ///< Attributes (bits [5:4] of byte 2) (default 0x0)
    PCIE_SI_EP,       ///< TLP is poisoned (bit 6 of byte 2) (default 0x0)
    PCIE_SI_TD,       ///< TLP digest present (bit 7 of byte 2). This bit will only be shown in the header and has no effect (the format of the TLP will not change) (default 0x0).
    PCIE_SI_RSVD013,  ///< Reserved (bits [3:0] of byte 1) (default 0x0)
    PCIE_SI_TC,       ///< Traffic Class (bits [6:4] of byte 1) (default 0x0)
    PCIE_SI_RSVD017,  ///< Reserved (bit 7 of byte 1) (default 0x0)
    PCIE_SI_TYPE,     ///< Type of TLP (bits 4:0 of byte 0) (default 0x0)
    PCIE_SI_FMT,      ///< Format of TLP (bits 6:5 of byte 0), possible property values: \ref EPCIEPktValFmt (default PCIE_PKTV_FMT_3DW)
    PCIE_SI_RSVD007,  ///< Reserved (bit 7 of byte 0) (default 0x0)
    PCIE_SI_1STDWBE,  ///< byte enables for the first (or only) DWORD (bits [3:0] of byte 7) (default 0xF)
    PCIE_SI_LASTDWBE, ///< byte enables for the last DWORD (bits [7:4] of byte 7) (default 0xF)
    PCIE_SI_TAG,      ///< Tag of the transaction (bits [7:0] of byte 6) (default 0x0). Used only if PCIE_SI_AUTOTAG equals 0.
    PCIE_SI_REQID,    ///< Requester ID (bits [7:0] of byte 5, bits [7:0] of byte 4) (default 0x0)
    PCIE_SI_CFG_RSVD111,  // Reserved (bits [1:0] of byte 11) (default 0x0)
    PCIE_SI_CFG_REGNUM,   ///< Register Number (bits [7:2] of byte 11) (default 0x0)
    PCIE_SI_CFG_EXTREGNUM,///< Extended Register Number (bits [3:0] of bytes 10) (default 0x0)
    PCIE_SI_CFG_RSVD107,  ///< Reserved (bits [7:4] of byte 10) (default 0x0)
    PCIE_SI_CFG_FUNCNUM,  ///< Function Number (bits [2:0] of byte 9) (default 0x0)
    PCIE_SI_CFG_DEVNUM,   ///< Device Number (bits [7:3] of byte 9) (default 0x0)
    PCIE_SI_CFG_BUSNUM,   ///< Bus Number (bits [7:0] of byte 8) (default 0x0)
    PCIE_SI_MEM64_ADDRHI, ///< Address bits [63:32] in a 64-bit memory access (bytes 8,9,10,11) (default 0x0)
    PCIE_SI_MEM64_ADDRLO, ///< Address bits [31:0] in a 64-bit memory access (bytes 12,13,14,15). Please be aware that it is possible to set bits [1:0] to any value. According to the PCIE Express Specification these must be 0, any other value may lead to unwanted behaviour. (default 0x0)
    PCIE_SI_MEM64_RSVD151,///< Reserved (bits [1:0] of byte 15) (default 0x0)
    PCIE_SI_MEM32_ADDR,   ///< Address bits [31:0] in a 32-bit memory access (bytes 8,9,10,11). Please be aware that it is possible to set bits [1:0] to any value. According to the PCIE Express Specifications, these must be 0, any other value may lead to unwanted behaviour. (default 0x0)
    PCIE_SI_MEM32_RSVD111,// Reserved (bits [1:0] of byte 11) (default 0x0)
    PCIE_SI_MCODE,        ///< Message code (bits [7:0] of byte 7) (default 0xFF)
    PCIE_SI_IO_ADDR,          ///< Address bits [31:0] in a 32-bit IO access (default 0x0)
    PCIE_SI_IO_RSVD111,       // Reserved (bits [1:0] of byte 11) (default 0x0)    
    PCIE_SI_MSGAS_RSVD047,    // Reserved for MsgAS (bytes 4,5,6,7) (default 0x0)
    PCIE_SI_MSGAS_RSVD087,    // Reserved for MsgAS (bytes 8,9,10,11) (default 0x0)
    PCIE_SI_MSGAS_RSVD127,    // Reserved for MsgAS (bytes 12,13,14,15) (default 0x0)
    PCIE_SI_MSG_BYTES08_11,    // Fields dependent on type of message (bytes 8,9,10,11) (default 0x0)
    PCIE_SI_MSG_BYTES12_15,    // Fields dependent on type of message (bytes 12,13,14,15) (default 0x0)
    PCIE_SI_MSGASD_RSVD047,    // Reserved for MsgASD (bytes 4,5,6,7) (default 0x0)
    PCIE_SI_MSGASD_RSVD087,    // Reserved for MsgASD (bytes 8,9,10,11) (default 0x0)
    PCIE_SI_MSGASD_RSVD127,    // Reserved for MsgASD (bytes 12,13,14,15) (default 0x0)
    PCIE_SI_COMP_REQID,///< Requester ID of the completion (bytes 8,9). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_COMPLID,///< Completion ID of the completion (bytes 4,5). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_STATUS,///< Completion status of the completion (bits[7:5] of byte 6). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_BYTECOUNT,///< Bytecount of the completion (bits[3:0] of byte 6 and byte 7). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_BCM,///< BCM of the completion (bit 4 of byte 6). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_TAG,///< Tag of the completion (byte 10). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_RSVD117,///< Reserved bits of the completion (bit 7 of byte 11). To be used in 'SiCompGet' only !
    PCIE_SI_COMP_LOWERADDR,///< Lower address field of the completion (bits[6:0] of byte 11). To be used in 'SiCompGet' only !
    PCIE_SI_OBSOLETE_46,        // Type of TLP (composed of FMT and TYPE field), possible property values: \ref EPCIEPktValType (default PCIE_PKTV_TLPTYPE_CFGRD0)
    PCIE_SI_AUTOTAG,        ///< 1: Tag field in SI packets is automatically created (default). 0: PCIE_SI_TAG is used for tag field in SI packets.
    PCIE_SI_RESOURCE,       ///< Resource the SI packet is used with, possible property values: \ref EPCIESiValResource (default PCIE_SIV_RESOURCE_SIMEM)
    PCIE_SI_INTADDR,
    ///< Internal address in data memory for SI requests, if data
    /// memory is used as resource (see PCIE_SI_RESOURCE) (default 0x0). This must be 4-byte aligned.
    PCIE_SI_BUSERROR,
    ///< Bus error to be inserted for the SI packet. Please see \ref EPCIESiValBusError (default 0x0)
    PCIE_SI_FUNCTION_SELECTED 
    ///< Select the function for SI packet send. \ref EPCIEHwChannelFunction. (default PCIE_HWCHANNEL_FUNCTION_A)(*New in Gen3 Exerciser*)
} EPCIESi;


/// \brief "Send Immediate" bus error
///
/// Possible values of property PCIE_SI_BUSERROR (see \ref EPCIESi). These are
/// used to insert errors in a Send Immediate request packet
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIESiValBusError
{
  PCIE_SIV_BUSERROR_NONE = 0,                 ///< (0) No bus error will be inserted.
  PCIE_SIV_BUSERROR_INCORRECT_DISPARITY = 1,  ///< (1) The packet will have an incorrect disparity (default 0x0).
  PCIE_SIV_BUSERROR_INCORRECT_ECRC = 2,       ///< (2) Incorrect ECRC is inserted (default 0x0).
  PCIE_SIV_BUSERROR_WRONG_PAYLOAD_SIZE = 4,   ///< (4) Wrong payload size is inserted (default 0x0).
                                              ///This field can be enabled or disabled. If enabled (0x1), the value in the LEN field of the packet is increased or increased by 1 (depending on \ref PCIE_EXERCISER_WRONG_PAYLOAD_SIZE_BEH which can be controlled by \ref ExerciserSet) before sending the packet.
                                              ///ATTENTION: This feature is not supported for packets with payload size of 1024 DW.
  PCIE_SIV_BUSERROR_NULLIFIED_TLP = 8,        ///< (8) The TLP will be nullified.
  PCIE_SIV_BUSERROR_INSERT_TD = 16,           ///< (16) The TLP Digest is inserted in the packet (default 0x0).
                                              // PCIE_SIV_BUSERROR_INSERT_TD replaces PCIE_SIV_BUSERROR_WRONG_TLP_DIGEST.
  PCIE_SIV_BUSERROR_INCORRECT_LCRC = 32,      ///< (32) The packet will have an incorrect LCRC.
  PCIE_SIV_BUSERROR_REPLACE_SEQNO=64,         ///< (64) Replace packet's sequence number. Actual sequence number is calculated by correct sequence number plus offset specified with property PCIE_DLLPHY_SEQNO_OFFSET (see function DllPhySet(), enum \ref EPCIEDllPhy).
  PCIE_SIV_BUSERROR_REPLACE_STP=128,          ///< (128) Replace the Start TLP special symbol (K27.7) with an arbitrary symbol, which can be set with property PCIE_DLLPHY_STP_SYMBOL (see function DllPhySet(), enum \ref EPCIEDllPhy).
  PCIE_SIV_BUSERROR_REPLACE_END=256           ///< (256) Replace the END special symbol (K29.7) with an arbitrary symbol, which can be set with property PCIE_DLLPHY_END_SYMBOL (see function DllPhySet(), enum \ref EPCIEDllPhy).
} EPCIESiValBusError;


/// \brief "Send Immediate" packet resource.
///
/// Possible values of property PCIE_SI_RESOURCE (see \ref EPCIESi).
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIESiValResource
{
  PCIE_SIV_RESOURCE_SIMEM = 0,    ///< (0) For write requests, the SI packet payload is inserted from SI requester memory. For read requests, incoming data is stored in SI completion memory.
  PCIE_SIV_RESOURCE_DATAMEM = 1,  ///< (1) SI packet payload is taken from the data memory on a write request and written to it on a read request.
  PCIE_SIV_RESOURCE_GENERATOR = 2 ///< (2) SI packet payload is inserted from data generator on a write request.
} EPCIESiValResource;

/// \brief Packet format
///
/// Possible values of property PCIE_SI_FMT (see \ref EPCIESi).
/// Possible values of property PCIE_BLOCK_FMT (see \ref EPCIEBlock).
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIEPktValFmt 
{
  PCIE_PKTV_FMT_3DW   = 0,  ///< (0) 3 DWORD header, no data
  PCIE_PKTV_FMT_4DW   = 1,  ///< (1) 4 DWORD header, no data
  PCIE_PKTV_FMT_3DWD  = 2,  ///< (2) 3 DWORD header, with data
  PCIE_PKTV_FMT_4DWD  = 3   ///< (3) 4 DWORD header, with data
} EPCIEPktValFmt;

#ifndef COMPILING_DOC
/// \brief "Send Immediate" packet type.
///
/// Possible values of property PCIE_SI_TLPTYPE (see \ref EPCIESi).
/// Possible values of property PCIE_BLOCK_TLPTYPE (see \ref EPCIEBlock).
typedef DEV_MIDL_ENUM enum EPCIEPktValType 
{
  PCIE_PKTV_TLPTYPE_MRD32, ///< Memory request with 32-bit addressing
  PCIE_PKTV_TLPTYPE_MRD64, ///< Memory request with 64-bit addressing
  PCIE_PKTV_TLPTYPE_MRDLK32 ,///< Memory read request-locked with 32-bit addressing
  PCIE_PKTV_TLPTYPE_MRDLK64 ,///< Memory read request-locked with 64-bit addressing 
  PCIE_PKTV_TLPTYPE_MWR32 ,///< Memory write request with 32-bit addressing 
  PCIE_PKTV_TLPTYPE_MWR64 ,///< Memory write request with 64-bit addressing  
  PCIE_PKTV_TLPTYPE_IORD, ///< I/O read request 
  PCIE_PKTV_TLPTYPE_IOWR,///< I/O write request
  PCIE_PKTV_TLPTYPE_CFGRD0,///< Configuration read type 0
  PCIE_PKTV_TLPTYPE_CFGWR0,///< Configuration write type 0 
  PCIE_PKTV_TLPTYPE_CFGRD1,///< Configuration read type 1 
  PCIE_PKTV_TLPTYPE_CFGWR1,///< Configuration write type 1   
  PCIE_PKTV_TLPTYPE_MSG_ROUT_ROOT, ///< Message request - routed to root complex
  PCIE_PKTV_TLPTYPE_MSG_ROUT_ADDR, ///< Message request - routed by address
  PCIE_PKTV_TLPTYPE_MSG_ROUT_ID, ///< Message request - routed by ID
  PCIE_PKTV_TLPTYPE_MSG_BDCST_ROOT, ///< Message request - broadcast from root complex
  PCIE_PKTV_TLPTYPE_MSG_LOCAL, ///< Message request - local - terminate at receiver
  PCIE_PKTV_TLPTYPE_MSG_GATH_ROUT_ROOT, ///< Message request - gathered and routed to root complex
  PCIE_PKTV_TLPTYPE_MSG_RSVD_110, ///< Message request - reserved
  PCIE_PKTV_TLPTYPE_MSG_RSVD_111,///< Message request - reserved
  PCIE_PKTV_TLPTYPE_MSGD_ROUT_ROOT, ///< Message request - routed to root complex
  PCIE_PKTV_TLPTYPE_MSGD_ROUT_ADDR, ///< Message request - routed by address
  PCIE_PKTV_TLPTYPE_MSGD_ROUT_ID, ///< Message request - routed by ID
  PCIE_PKTV_TLPTYPE_MSGD_BDCST_ROOT, ///< Message request - broadcast from root complex
  PCIE_PKTV_TLPTYPE_MSGD_LOCAL, ///< Message request - local - terminate at receiver
  PCIE_PKTV_TLPTYPE_MSGD_GATH_ROUT_ROOT, ///< Message request - gathered and routed to root complex
  PCIE_PKTV_TLPTYPE_MSGD_RSVD_110, ///< Message request - reserved
  PCIE_PKTV_TLPTYPE_MSGD_RSVD_111,///< Message request - reserved 
  PCIE_PKTV_TLPTYPE_MSGAS_000, // Message for advanced switching request - reserved
  PCIE_PKTV_TLPTYPE_MSGAS_001,
  PCIE_PKTV_TLPTYPE_MSGAS_010,
  PCIE_PKTV_TLPTYPE_MSGAS_011,
  PCIE_PKTV_TLPTYPE_MSGAS_100,
  PCIE_PKTV_TLPTYPE_MSGAS_101,
  PCIE_PKTV_TLPTYPE_MSGAS_110,
  PCIE_PKTV_TLPTYPE_MSGAS_111,
  PCIE_PKTV_TLPTYPE_MSGASD_000, // Message for advanced switching request - reserved
  PCIE_PKTV_TLPTYPE_MSGASD_001,
  PCIE_PKTV_TLPTYPE_MSGASD_010,
  PCIE_PKTV_TLPTYPE_MSGASD_011,
  PCIE_PKTV_TLPTYPE_MSGASD_100,
  PCIE_PKTV_TLPTYPE_MSGASD_101,
  PCIE_PKTV_TLPTYPE_MSGASD_110,
  PCIE_PKTV_TLPTYPE_MSGASD_111,  
  PCIE_PKTV_TLPTYPE_CPL, ///< Completion without data
  PCIE_PKTV_TLPTYPE_CPLD, ///< Completion with data
  PCIE_PKTV_TLPTYPE_CPLLK, ///< Completion for locked memory read without data
  PCIE_PKTV_TLPTYPE_CPLDLK ///< Completion for locked memory read with data  
} EPCIEPktValType;      // TODO (DOC): changed according the first declaration! please change all occurrances if needed (used in doxygen links as well)
#endif

/// \brief Available hardware channels
///
typedef DEV_MIDL_ENUM enum EPCIEHwChannelFunction 
{
  PCIE_HWCHANNEL_FUNCTION_A,    ///< (0) Hardware Channel of Function A
  PCIE_HWCHANNEL_FUNCTION_B,    ///< (1) Hardware Channel of Function B
  PCIE_HWCHANNEL_FUNCTION_C,    ///< (2) Hardware Channel of Function C
  PCIE_HWCHANNEL_FUNCTION_D,    ///< (3) Hardware Channel of Function D (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_E,    ///< (4) Hardware Channel of Function E (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_BVF1, ///< (5) Hardware Channel of virtual function VF1 associated with Function B. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*) 
  PCIE_HWCHANNEL_FUNCTION_BVF2, ///< (6) Hardware Channel of virtual function VF2 associated with Function B. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_CVF1, ///< (7) Hardware Channel of virtual function VF1 associated with Function C. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_CVF2, ///< (8) Hardware Channel of virtual function VF2 associated with Function C. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_DVF1, ///< (9) Hardware Channel of virtual function VF1 associated with Function D. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_DVF2, ///< (10) Hardware Channel of virtual function VF2 associated with Function D. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_EVF1, ///< (11) Hardware Channel of virtual function VF1 associated with Function E. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_EVF2, ///< (12) Hardware Channel of virtual function VF2 associated with Function E. Valid for MRIOV/SRIOV mode only. (*New in Gen3 Exerciser*)
  PCIE_HWCHANNEL_FUNCTION_END
} EPCIEHwChannelFunction;

/// \brief Function Table Type
///
typedef DEV_MIDL_ENUM enum EFTPropType
{
  FT_INT_MAP, ///< (0) Function Interrupt Map
  FT_VH0,     ///< (1) Function Table for VH0
  FT_VH1,     ///< (2) Function Table for VH1
  FT_VH2,     ///< (3) Function Table for VH2
  FT_VH3,     ///< (4) Function Table for VH3
  FT_VH4,     ///< (5) Function Table for VH4
}EFTPropType;

/// \brief Available VC Resources  (*New in Gen3 Exerciser*)
///
typedef DEV_MIDL_ENUM enum EPCIEVCResourceId
{
  PCIE_VC_RESOURCE_0, ///<VC Resource 0. VC0 in PCIe & SRIOV Mode, VH0 in MRIOV Mode
  PCIE_VC_RESOURCE_1, ///<VC Resource 1. VC[x] in PCIe & SRIOV Mode, VH1 in MRIOV Mode
  PCIE_VC_RESOURCE_2, ///<VC Resource 2. (*New in Rel3 Exerciser, Applicable only in MRIOV [VH2]*)
  PCIE_VC_RESOURCE_3, ///<VC Resource 3. (*New in Rel3 Exerciser, Applicable only in MRIOV [VH3]*)
  PCIE_VC_RESOURCE_4, ///<VC Resource 4. (*New in Rel3 Exerciser, Applicable only in MRIOV [VH4]*)
  PCIE_VL_RESOURCE_0, ///<VL Resource 0. (*New in Rel3 Exerciser, Applicable only in MRIOV [VL0]*)
  PCIE_VC_RESOURCE_END,
}EPCIEVCResourceId;

/// \brief Hardware channel properties
///
/// Properties for the Hardware Channels. For the meaning of the initially
/// advertised credit limits, please refer to the PCI Express specification.
typedef DEV_MIDL_ENUM enum EPCIEVCResourceProp
{
  PCIE_HWCHANNELP_VCID,
  ///< Virtual Channel ID of this VC Resource. VC ID for VC Resource 0:'0'(cannot be changed), 
  ///VC ID for VC Resource 1: 0-7
  PCIE_HWCHANNELP_VC_ENABLE,     ///< Enable the Virtual Channel Resource. Valid values are: 0 to disable the resource and 1 to enable the resource (*New in Gen3 Exerciser*)
  PCIE_HWCHANNELP_OBSOLETE_01,
  PCIE_HWCHANNELP_IGNORE_CLIMIT, ///< Credit limit of link partner is ignored. There are two possible values: 0 (Credit limit is not ignored) and 1 (Credit limit is ignored). Default is 0.
  PCIE_HWCHANNELP_INITADV_PH,    ///< Initially advertised PH credits (default is 0x40) 
  PCIE_HWCHANNELP_INITADV_PD,    ///< Initially advertised PD credits (default is 0x400)
  PCIE_HWCHANNELP_INITADV_NPH,   ///< Initially advertised NPH credits (default is 0x40)
  PCIE_HWCHANNELP_INITADV_NPD,   ///< Initially advertised NPD credits  (default is 0x400)
  PCIE_HWCHANNELP_INITADV_CPLH,  ///< Initially advertised CPLH credits (default is 0x7F)
  PCIE_HWCHANNELP_INITADV_CPLD   ///< Initially advertised CPLD credits (default is 0x0 (infinite))
} EPCIEVCResourceProp;


/// \brief Virtual channel status properties
///
/// Provides information on the status of the hardware channels. Tnis is only available for VC
/// Resource 0 and 1.
///
/// The following Three types of values can be read, each for the different credit limit
/// counters:
/// -# Allocated:
///    This value indicates the credits allocated by the link partner and indicates its buffer size.
/// -# Limit:
///    This value indicates the credit limit of the Exerciser. The credit limit counter must be larger
///    than the Consumed counter. Otherwise, there are no credits available to send a packet. Each time the 
///    link partner frees buffer (via a flow control update packet), the credit limit of Exerciser increases.
/// -# Consumed:
///    This value indicates the credits already consumed by the Exerciser. Each time a packet is sent, 
///    this counter increases. It cannot exceed the Limit counter.
typedef DEV_MIDL_ENUM enum EPCIEVCResourceStatusProp
{
    PCIE_HWCHANNELSTATUSP_ALLOC_PH,    ///< Allocated posted request headers
    PCIE_HWCHANNELSTATUSP_ALLOC_PD,    ///< Allocated posted request data payload
    PCIE_HWCHANNELSTATUSP_ALLOC_NPH,   ///< Allocated non-posted request headers
    PCIE_HWCHANNELSTATUSP_ALLOC_NPD,   ///< Allocated non-posted request data payload
    PCIE_HWCHANNELSTATUSP_ALLOC_CPLH,  ///< Allocated completion headers
    PCIE_HWCHANNELSTATUSP_ALLOC_CPLD,   ///< Allocated completion data payload
    PCIE_HWCHANNELSTATUSP_LIMIT_PH,    ///< Limit for posted request headers
    PCIE_HWCHANNELSTATUSP_LIMIT_PD,    ///< Limit for posted request data payload
    PCIE_HWCHANNELSTATUSP_LIMIT_NPH,   ///< Limit for non-posted request headers
    PCIE_HWCHANNELSTATUSP_LIMIT_NPD,   ///< Limit for non-posted request data payload
    PCIE_HWCHANNELSTATUSP_LIMIT_CPLH,  ///< Limit for completion headers
    PCIE_HWCHANNELSTATUSP_LIMIT_CPLD,   ///< Limit for completion data payload
    PCIE_HWCHANNELSTATUSP_CONS_PH,     ///< Consumed posted request headers
    PCIE_HWCHANNELSTATUSP_CONS_PD,     ///< Consumed posted request data payload
    PCIE_HWCHANNELSTATUSP_CONS_NPH,    ///< Consumed non-posted request headers
    PCIE_HWCHANNELSTATUSP_CONS_NPD,    ///< Consumed non-posted request data payload
    PCIE_HWCHANNELSTATUSP_CONS_CPLH,   ///< Consumed completion headers
    PCIE_HWCHANNELSTATUSP_CONS_CPLD    ///< Consumed completion data payload
} EPCIEVCResourceStatusProp; 



/// \brief Link State Values
///
/// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIELTSSMVal
{
  PCIE_LTSSMV_IDLE                           = 0, ///< (0) The LTSSM is idle. This is the initial state of the LTSSM. To initiate the link training, please do a \ref LinkUp.
  PCIE_LTSSMV_DETECT_QUIET                   = 1, ///< (1) Detect.Quiet
  PCIE_LTSSMV_DETECT_ACTIVE                  = 2, ///< (2) Detect.Active 
  PCIE_LTSSMV_POLLING_ACTIVE                 = 3, ///< (3) Polling.Active
  PCIE_LTSSMV_POLLING_CONFIG                 = 4, ///< (4) Polling.Configuration
  PCIE_LTSSMV_CONFIG_LINKWIDTH_START         = 5, ///< (5) Config.LinkWidthStart
  PCIE_LTSSMV_CONFIG_LINKWIDTH_ACCEPT        = 6, ///< (6) Config.LinkWidthAccept
  PCIE_LTSSMV_CONFIG_LANENUM_WAIT            = 7, ///< (7) Config.LaneNumWait
  PCIE_LTSSMV_CONFIG_LANENUM_ACCEPT          = 8, ///< (8) Config.LaneNumAccept
  PCIE_LTSSMV_CONFIG_COMPLETE                = 9, ///< (9) Config.Complete
  PCIE_LTSSMV_CONFIG_IDLE                    =10, ///< (10) Config.Idle
  PCIE_LTSSMV_RECOVERY_RCVRLOCK              =11, ///< (11) Recovery.RcvrLock
  PCIE_LTSSMV_RECOVERY_RCVRCFG               =12, ///< (12) Recovery.RcvrCfg
  PCIE_LTSSMV_RECOVERY_SPEED                 =13, ///< (13) Recovery.Speed
  PCIE_LTSSMV_RECOVERY_IDLE                  =14, ///< (14) Recovery.Idle
  PCIE_LTSSMV_L1_ENTRY                       =15, ///< (15) L1.Entry
  PCIE_LTSSMV_L1_IDLE                        =16, ///< (16) L1.Idle
  PCIE_LTSSMV_DISABLED                       =17, ///< (17) Disabled
  PCIE_LTSSMV_LOOPBACK_ENTRY                 =18, ///< (18) Loopback.Entry
  PCIE_LTSSMV_LOOPBACK_ACTIVE                =19, ///< (19) Loopback.Active
  PCIE_LTSSMV_LOOPBACK_EXIT                  =20, ///< (20) Loopback.Exit
  PCIE_LTSSMV_HOT_RESET                      =21, ///< (21) HotReset
  PCIE_LTSSMV_ERR_STATE_1                    =22, ///< (22) Error State 1
  PCIE_LTSSMV_ERR_STATE_2                    =23, ///< (23) Error State 2
  PCIE_LTSSMV_TX_L0_RX_LO                    =24, ///< (24) Transmitter and receiver are in state L0
  PCIE_LTSSMV_TX_L0_RX_L0S                   =25, ///< (25) Transmitter is in state L0 and receiver is in state L0s. Writing this value with the LTSSM LA feature has the same effect as writing PCIE_LTSSMV_TX_L0_RX_LO.
  PCIE_LTSSMV_TX_L0S_RXL0                    =26, ///< (26) Transmitter is in state L0s and receiver is in state L0
  PCIE_LTSSMV_TX_L0S_RXL0S                   =27, ///< (27) Transmitter and receiver are in state L0s. Writing this value with the LTSSM LA feature has the same effect as writing PCIE_LTSSMV_TX_L0S_RXL0.
  PCIE_LTSSMV_INVALID                        =28, ///< (28) Unable to interpret state information returned by FPGA. This enum is read-only.
  PCIE_LTSSMV_MASK_ALL                       =29, ///< (29) Only to be used with the LTSSM LA feature. This sets the trigger_mask, trigger_level or change_mask to all '1'.
  PCIE_LTSSMV_UNMASK_ALL                     =30, ///< (30) Only to be used with the LTSSM LA feature. This sets the trigger_mask, trigger_level or change_mask to all '0'.
  PCIE_LTSSMV_RECOVERY_EQUALIZATION_PHASE0   =31, ///< (31) Equalization.Phase0. (*New in Gen3 Exerciser*)
  PCIE_LTSSMV_RECOVERY_EQUALIZATION_PHASE1   =32, ///< (32) Equalization.Phase1. (*New in Gen3 Exerciser*)
  PCIE_LTSSMV_RECOVERY_EQUALIZATION_PHASE2   =33, ///< (33) Equalization.Phase2. (*New in Gen3 Exerciser*)
  PCIE_LTSSMV_RECOVERY_EQUALIZATION_PHASE3   =34, ///< (34) Equalization.Phase3. (*New in Gen3 Exerciser*)
} EPCIELTSSMVal;

#ifndef COMPILING_DOC

/// \brief Tx Status Values
///
/// Tx Status Values
/// RC is 'Read Clear' - bits are cleared on status read
///
typedef DEV_MIDL_ENUM enum EPCIETxStatus 
{
  PCIE_TXSTATUS_DBI_LEFT_RIGHT_DETECT,  ///< DBI left/right detect
    PCIE_TXSTATUS_CLOCKS_READY,///< clocks ready (RO), '1' when all DCMs are currently locked
    PCIE_TXSTATUS_REQHEADER_GENA_RUNNING,
    PCIE_TXSTATUS_REQHEADER_GENA_IDLE,
    PCIE_TXSTATUS_REQHEADER_GENB_RUNNING,
    PCIE_TXSTATUS_REQHEADER_GENB_IDLE,
    PCIE_TXSTATUS_REQHEADER_GENC_RUNNING,
    PCIE_TXSTATUS_REQHEADER_GENC_IDLE,
		PCIE_TXSTATUS_REQHEADER_GEND_RUNNING,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GEND_IDLE,       ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENE_RUNNING,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENE_IDLE,       ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENBVF1_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENBVF1_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENBVF2_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENBVF2_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENCVF1_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENCVF1_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENCVF2_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENCVF2_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENDVF1_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENDVF1_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENDVF2_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENDVF2_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENEVF1_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENEVF1_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENEVF2_RUNNING, ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_REQHEADER_GENEVF2_IDLE,    ///< (*New in Gen3 Exerciser*)
    PCIE_TXSTATUS_SI_SENT,///< send immediate packet sent (RC)
    PCIE_TXSTATUS_COMP_QUEUE0_OVERFLOW,///< Completion queue 0 overflow (RC)
    PCIE_TXSTATUS_COMP_QUEUE1_OVERFLOW,///< Completion queue 1 overflow (RC)
		PCIE_TXSTATUS_COMP_QUEUE2_OVERFLOW,///< Completion queue 2 overflow (RC)
    PCIE_TXSTATUS_COMP_QUEUE3_OVERFLOW,///< Completion queue 3 overflow (RC)
    PCIE_TXSTATUS_COMP_QUEUE4_OVERFLOW,///< Completion queue 4 overflow (RC)
    PCIE_TXSTATUS_INTTEST///< Interrupt Test (RC)
} EPCIETxStatus;


/// \brief Rx Status Values
/// RC is 'Read Clear' - bits are cleared on status read
///
typedef DEV_MIDL_ENUM enum EPCIERxStatus 
{
  PCIE_RXSTATUS_DBI_LEFT_RIGHT_DETECT = 0,///< (0) left/right detect. This is obsolete for Gen3.
    PCIE_RXSTATUS_CLOCKS_READY = 1,       ///< (1) clocks ready. This is obsolete for Gen3.
    PCIE_RXSTATUS_RX_RUNNING = 2,         ///< (2) RX running
    PCIE_RXSTATUS_DLLPHY_ERROR = 3,       ///< (3) DLL/PHY error (RC)
    PCIE_RXSTATUS_PATTERN_MATCHER = 4,    ///< (4) pattern matcher (RC)
    PCIE_RXSTATUS_INVALID_REQUEST = 5,    ///< (5) invalid request (RC) (not implemented)
    PCIE_RXSTATUS_MARKED_COMPLETION = 6,  ///< (6) marked completion (RC) (not implemented)
    PCIE_RXSTATUS_FC_INIT_COMPLETE = 7,   ///< (7) FC initialization complete (RC)
    PCIE_RXSTATUS_DATA_MEM_HOT_READ_COMPLETE = 8,///< (8) data memory hot read complete (RC) (not implemented)
    PCIE_RXSTATUS_RCVED_CONFIG_REQUEST = 9,///< (9) received configuration request (RC)
    PCIE_RXSTATUS_RCVED_MESSAGE = 10,       ///< (10) received message (RC)
    PCIE_RXSTATUS_DEC_MISS = 11,            ///< (11) decoder miss (RC)
    PCIE_RXSTATUS_FIFO_OVERFLOW = 12,       ///< (12) PRM new entry FIFO overflow (RC)
    PCIE_RXSTATUS_RX_TLP_ON_DISABLED_VC = 13,///< (13) RX TLP on disabled VC (RC) (not implemented)
    PCIE_RXSTATUS_DATA_WHILE_DISABLED = 14,  ///< (14) data memory controller; incoming data while disabled; data lost (RC)
    PCIE_RXSTATUS_SIMEM_NEW_COMP = 15,       ///< (15) SI completion memory = 0 new completion (RC)
    PCIE_RXSTATUS_SIMEM_OVERFLOW = 16,       ///< (16) SI completion memory = 0 overflow (RC)
    PCIE_RXSTATUS_SIMEM_OVERRUN = 17,        ///< (17) SI completion memory = 0 overrun (RC)
    PCIE_RXSTATUS_MSG_AVAILABLE = 18,        ///< (18) msg_status: message available
    PCIE_RXSTATUS_CFG_ACCESS_PENDING = 19,   ///< (19) Config access pending
    PCIE_RXSTATUS_SI_COMP_AVAILABLE = 20,    ///< (20) SI completion packet available
    PCIE_RXSTATUS_SW_PACKET_OVERFLOW = 21,   ///< (21) overflow (RC)
    PCIE_RXSTATUS_SW_PACKET_DATA_AVAILABLE = 22,///< (22) data available (RC)
    PCIE_RXSTATUS_DL_DOWN = 23,                 ///< (23) The Data link layer (DLL) is not communicating with the component on the other side of the link (RC)
    PCIE_RXSTATUS_DL_UP = 24                    ///< (24) The Data link layer (DLL) is communicating with the component on the other side of the link (RC)
} EPCIERxStatus;

/// \brief Statistics Status
///
typedef DEV_MIDL_ENUM enum EPCIEStatisticsStatus
{
   PCIE_STATISTICSSTATUS_CFG_RCVED, ///< Total number of received config access packets
   PCIE_STATISTICSSTATUS_CFG_RESP,   ///< Total number of config access completions sent
   PCIE_STATISTICSSTATUS_CFG_RCVED_READ0, ///< Total number of config read type 0 accesses
   PCIE_STATISTICSSTATUS_CFG_RCVED_WRITE0, ///< Total number of config write type 0 accesses
   PCIE_STATISTICSSTATUS_CFG_RCVED_READ1, ///< Total number of config read type 1 accesses
   PCIE_STATISTICSSTATUS_CFG_RCVED_WRITE1, ///< Total number of config write type 0 accesses
   PCIE_STATISTICSSTATUS_CFG_ACCESS_START_TIME,
   PCIE_STATISTICSSTATUS_CFG_ACCESS_STOP_TIME
} EPCIEStatisticsStatus;

/// \brief Data Error Status
/// RC is 'Read Clear' - bits are cleared on status read
///
typedef DEV_MIDL_ENUM enum EPCIEDataErrorStatus 
{
  PCIE_DATAERRSTATUS_GENERATOR_MISMATCH = 0,///< (0) Data analyzer mismatch (RC)
  PCIE_DATAERRSTATUS_MEM_MISMATCH = 1///< (1) Exerciser memory mismatch (RC)
} EPCIEDataErrorStatus;

// For compatibility only
/// \brief Protocol Error Status
/// RC is 'Read Clear' - bits are cleared on status read
///
/// Using ProtocolErrorStatusRead, you receive a 32bit value. The number parentheses indicates the bit position of the information.
typedef DEV_MIDL_ENUM enum EPCIEProtocolErrorStatus 
{
  PCIE_PROTERRSTATUS_UNEXPECTED_CRS_COMP = 0,          //  (0) unexpected CRS completion (RC) (not implemented)
    PCIE_PROTERRSTATUS_COMP_OTHER_THAN_SUCC_OR_CRS = 1,//  (1) completion status other than (successful OR CRS) (RC) CAUTION: NOT implemented in HW (replaced by TriggerOut INT bit)
    PCIE_PROTERRSTATUS_ECRC_ERROR_PRESENT = 2,         //  (2) ECRC error occured (RC) (not implemented)
    PCIE_PROTERRSTATUS_ECRC_ERROR_VAL = 3,             //  (3) ECRC error, value (RC) (not implemented)
    PCIE_PROTERRSTATUS_UNEXPECTED_COMP = 4,            ///<  (4) unexpected completion (RC). ATTENTION: This enum entry is obsolete for Gen3. Please use \ref ProtocolErrorFirstRead or \ref ProtocolErrorAccuRead with property PCIE_PROTRULE_UNEXPECTED_COMPLETION
    PCIE_PROTERRSTATUS_COMP_TIMEOUT = 5,               ///<  (5) completion timeout (RC). ATTENTION: This enum entry is obsolete for Gen3. Please use \ref ProtocolErrorFirstRead or \ref ProtocolErrorAccuRead with property PCIE_PROTRULE_COMPLETION_TIMEOUT
    PCIE_PROTERRSTATUS_UNSUPPORTED_REQUEST = 6,        ///<  (6) unsupported request (RC). ATTENTION: This enum entry is obsolete for Gen3. Please use \ref ProtocolErrorFirstRead or \ref ProtocolErrorAccuRead with property PCIE_PROTRULE_UNSUPPORTED_REQUEST
    PCIE_PROTERRSTATUS_FC_UPDATE_OVERRUN = 7,          ///<  (7) FC Update Overrun occured
    PCIE_PROTERRSTATUS_VC0_FC_PASSED = 8, ///<  (8) VC 0 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC1_FC_PASSED = 9, ///<  (9) VC 1 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC2_FC_PASSED = 10,///< (10) VC 2 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC3_FC_PASSED = 11,///< (11) VC 3 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC4_FC_PASSED = 12,///< (12) VC 4 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC5_FC_PASSED = 13,///< (13) VC 5 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC6_FC_PASSED = 14,///< (14) VC 6 has passed flow control initialization
    PCIE_PROTERRSTATUS_VC7_FC_PASSED = 15,///< (15) VC 7 has passed flow control initialization
    PCIE_PROTERRSTATUS_LCRC_ERROR_PRESENT = 16, // (16) LCRCR error occured (RC)
    PCIE_PROTERRSTATUS_TLP_OUT_OF_SEQUENCE = 17,// (17) Sequence number of TLP was out of sequence (RC)
    PCIE_PROTERRSTATUS_DLLP_WITH_BAD_EBD = 18,  // (18) DLLP with bad EBD (RC)
    PCIE_PROTERRSTATUS_RESERVED_DLP_TYPE = 19   // (19) Reserved DLP Type (RC)
} EPCIEProtocolErrorStatus;

#endif /* COMPILING_DOC */

/// \brief Protocol Rules 
///
typedef DEV_MIDL_ENUM enum EPCIEProtRule
{
  /****************************************************************************************************
   *** Keep number of entries consistent with PCIE_NUM_OF_PROTOCOL_RULES from devpcieintdefs.h 
   ****************************************************************************************************/
 
  /****************************************************************************************************/
  /****** Please keep rule descriptions consistent within these files:

      \ipcommon\shared\dev\devpciedefs.h (docu in enum EPCIEProtRule)
      \ipmodel\host\system\sessions\PCIExpress\Pciapi\AgtPCIEExerciser.cpp (in-system port API)
      \ipmodel\host\system\subsystems\pciexpress\AgtPCIEExerciser.cpp (Exerciser Interface implementation)
 
   ****************************************************************************************************/
  //TBD:GEN3 Need to define FPGA Register bit postion in seperate constants
  
  PCIE_PROTRULE_UNSUPPORTED_REQUEST   =0,  ///< The type of an incoming request is not supported. An unsupported request completion is generated (currently only for MsgASD). See PCI Express 1.0a, section 2.3.1, "If the Request Type is not supported by the device, the Request is an Unsupported Request and is reported ..."
  PCIE_PROTRULE_DECODER_MISS          =1,  ///< Received request packet not covered by any decoder. If subtractive decode is enabled, no decoder miss can occur. In this case, decoder misses are caught by the subtractive decoder.
  PCIE_PROTRULE_NULLIFIED_LCRC        =2,  ///< Received packet with nullified LCRC
  PCIE_PROTRULE_BAD_EDB_LCRC          =3,  ///< TLP with bad end EDB or bad (not poisoned) LCRC.
  PCIE_PROTRULE_NAK_SENT              =4,  ///< NAK sent
  PCIE_PROTRULE_TLP_SEQUENCE          =5,  ///< Received an out-of-sequence TLP. The protocol check finds two different incorrect behaviors:
                                           ///- The Exerciser receives an ACK or NAK with sequence number A. After that, it receives an ACK or NAK with a predecessor of A. A sequence number X is a predecessor of A if [(X-A) MOD 2^12 > 2^11].
                                           ///- The Exerciser receives an ACK or NAK with sequence number A and the TLP with sequence number A has not yet been sent or has been sent far in the past. If X is the sequence number of the last transmitted TLP, then A is an incorrect sequence number if [(X-A) MOD 2^12 > 2^11].
  PCIE_PROTRULE_DLLP_CRC              =7,  ///< DLLP checksum error
  PCIE_PROTRULE_DESKEW_ALIGN          =8 , ///< Malformed TLP
  PCIE_PROTRULE_DISPARITY             =9,  ///< Running disparity error at Gen1/Gen2 data rate
  PCIE_PROTRULE_CODING                =10, ///< Coding (symbol) error at Gen1/Gen2 Data rate.
  PCIE_PROTRULE_128B_130B_SYNC_ERROR  =11, ///< Block allignment is not established at Gen3 data rate.
  //PCIE_PROTRULE_DLLP_EBD              =6,  ///< Received a DLLP with a bad end (EBD).(Not supported in Gen3)   
	PCIE_PROTRULE_PCIE_ECRC_ERROR       =28, ///< ECRC error in pcie mode. (*New in Gen3 Exerciser*)
  PCIE_PROTRULE_END
} EPCIEProtRule;

/// \brief Protocol Rules reported for each VC Resource 
///
typedef DEV_MIDL_ENUM enum EPCIEProtRuleVC
{
  /****************************************************************************************************
   *** Keep number of entries consistent with PCIE_NUM_OF_PROTOCOL_RULES from devpcieintdefs.h 
   ****************************************************************************************************/
 
  /****************************************************************************************************/
  /****** Please keep rule descriptions consistent within these files:

      \ipcommon\shared\dev\devpciedefs.h (docu in enum EPCIEProtRule)
      \ipmodel\host\system\sessions\PCIExpress\Pciapi\AgtPCIEExerciser.cpp (in-system port API)
      \ipmodel\host\system\subsystems\pciexpress\AgtPCIEExerciser.cpp (Exerciser Interface implementation)
 
   ****************************************************************************************************/

  PCIE_PROTRULE_FC_TIMEOUT,  ///< Flow control update packet not received. PCI Express, section 2.6.1.2, sub-section "Flow Control Update Frequency"
	PCIE_PROTRULE_MRIOV_ECRC_ERROR,  ///< ECRC error in mriov mode. (*New in Gen3 Exerciser*)
  PCIE_PROTRULE_VC_END
} EPCIEProtRuleVC;

/// \brief Protocol Rules reported for each Function 
///
typedef DEV_MIDL_ENUM enum EPCIEProtRuleFunction
{
  /****************************************************************************************************
   *** Keep number of entries consistent with PCIE_NUM_OF_PROTOCOL_RULES from devpcieintdefs.h 
   ****************************************************************************************************/
 
  /****************************************************************************************************/
  /****** Please keep rule descriptions consistent within these files:

      \ipcommon\shared\dev\devpciedefs.h (docu in enum EPCIEProtRule)
      \ipmodel\host\system\sessions\PCIExpress\Pciapi\AgtPCIEExerciser.cpp (in-system port API)
      \ipmodel\host\system\subsystems\pciexpress\AgtPCIEExerciser.cpp (Exerciser Interface implementation)
 
   ****************************************************************************************************/
  PCIE_PROTRULE_COMPLETION_DATA ,  ///< The last completion for a request contains more data than expected. The tag from the incoming completion is blocked after that. It is not used anymore by the Exerciser until the tags are initialized by tag reset or link training.
  PCIE_PROTRULE_UNEXPECTED_COMPLETION  ,  ///< Received an unexpected completion packet that is, a completion packet that cannot associate with any outstanding request. The tag from the incoming completion is blocked after that. It is not used anymore by the Exerciser until the tags are initialized by tag reset or link training.
  PCIE_PROTRULE_COMPLETION_TIMEOUT     ,  ///< The Exerciser has detected that it has not received an outstanding completion within a programmable time. It is not guaranteed that the Exerciser detects a completion timeout immediately.
  //PCIE_PROTRULE_TLP_LENGTH           ,  ///< Malformed TLP: Packet length in header and actual payload length do not match. PCIE 1.0a, section 2.2.2: "For TLPs, that include data, the value in the Length field and the actual amount of data included in the TLP must match. Receivers must check for violations of this rule. If a Receiver determines that a TLP violates this rule, the TLP is a Malformed TLP." PCIE 1.0a, section 2.2.3: "For any TLP, a value of 1b in the TD field indicates the presence of the TLP Digest field including an ECRC value at the end of the TLP. A TLP where the TD field value does not correspond with the observed size (accounting for the data payload, if present) is a Malformed TLP.".
  PCIE_PROTRULE_FUNCTION_END
} EPCIEProtRuleFunction;

/// \brief Data Link State
///
/// The return value of \ref DataLinkStateRead will be the number indicated in parentheses.
typedef DEV_MIDL_ENUM enum EPCIEDataLinkState
{
  PCIE_DATALINKSTATE_DL_INACTIVE = 0, ///< (0) Physical Layer reporting link is non-operational or nothing is connected to the port
  PCIE_DATALINKSTATE_DL_INIT = 1,     ///< (1) Physical Layer reporting link is operational. Initialize Flow Control for the default VC
  PCIE_DATALINKSTATE_DL_ACTIVE = 2    ///< (2) Normal operation mode
} EPCIEDataLinkState;

/// \brief Block Properties
///
/// Properties of a TLP packet in the block memory.
/// For properties that are represented in the TLP, you will find additional information about the location in the header.
/// For example: "bits[3:2] of byte 2" means: This value will appear in bits 2 und 3 of byte 2 in the TLP header.
/// ATTENTION: It is not possible to set the Requester ID on a per line basis. Please use \ref ExerciserSet to program the Requester ID.
/// ATTENTION: Bytes15:8 of a message packet cannot be modified and are 0x0.
typedef DEV_MIDL_ENUM enum EPCIEBlock
{
  PCIE_BLOCK_LEN = 0,  ///< Total Length of data payload in DWORD (max. 1024 DWORDs). Please adapt 'PCIE_BLOCK_LASTDWBE' accordingly when modifying this value. (default 0x1)
  PCIE_BLOCK_RSVD023,  ///< Reserved (bits [3:2] of byte 2) (default 0x0)
  PCIE_BLOCK_ATTR,     ///< Attributes (bits [5:4] of byte 2) (default 0x0)
  PCIE_BLOCK_RSVD013,  ///< Reserved (bits [3:0] of byte 1) (default 0x0)
  PCIE_BLOCK_TC,       ///< Traffic Class (bits [6:4] of byte 1) (default 0x0)
  PCIE_BLOCK_RSVD017,  ///< Reserved (bit 7 of byte 1) (default 0x0)
  PCIE_BLOCK_TYPE,     ///< Type of TLP (bits 4:0 of byte 0) (default 0x0)
  PCIE_BLOCK_FMT,      ///< Format of TLP (bits 6:5 of byte 0), possible property values: \ref EPCIEPktValFmt, (default PCIE_PKTV_FMT_3DW)
  PCIE_BLOCK_RSVD007,  ///< Reserved (bit 7 of byte 0) (default 0x0)
  PCIE_BLOCK_1STDWBE,  ///< byte enables for the first (or only) DWORD (bits [3:0] of byte 7) (default 0xF)
  PCIE_BLOCK_LASTDWBE, ///< byte enables for the last DWORD (bits [7:4] of byte 7) (default 0xF)
  PCIE_BLOCK_CFG_RSVD111,  // Reserved (bits [1:0] of byte 11) (default 0x0)
  PCIE_BLOCK_CFG_REGNUM,   ///< Register Number (bits [7:2] of byte 11) (default 0x0)
  PCIE_BLOCK_CFG_EXTREGNUM,///< Extended Register Number (bits [3:0] of byte 10) (default 0x0)
  PCIE_BLOCK_CFG_RSVD107,  ///< Reserved (bits [7:4] of byte 10) (default 0x0)
  PCIE_BLOCK_CFG_FUNCNUM,  ///< Function Number (bits [2:0] of byte 9) (default 0x0)
  PCIE_BLOCK_CFG_DEVNUM,   ///< Device Number (bits [7:3] of byte 9) (default 0x0)
  PCIE_BLOCK_CFG_BUSNUM,   ///< Bus Number (bits [7:0] of byte 8) (default 0x0)
  PCIE_BLOCK_MEM64_ADDRHI, ///< Address bits [63:32] in a 64-bit memory access (bytes 8,9,10,11) (default 0x0)
  PCIE_BLOCK_MEM64_ADDRLO, ///< Address bits [31:0] in a 64-bit memory access (bytes 12,13,14,15). Please be aware that it is not possible to set bits [1:0]. These bits are reserved and will always be zero. Please use the SI packet functionality if it is necessary to set these bits. (default 0x0)
  PCIE_BLOCK_MEM64_RSVD151,// Reserved (bits [1:0] of byte 15) (default 0x0)
  PCIE_BLOCK_MEM32_ADDR,   ///< Address bits [31:0] in a 32-bit memory access (bytes 8,9,10,11). Please be aware that it is not possible to set bits [1:0]. These bits are reserved and will always be zero. Please use the SI packet functionality if it is necessary to set these bits. (default 0x0)
  PCIE_BLOCK_MEM32_RSVD111,// Reserved (bits [1:0] of byte 11) (default 0x0)
  PCIE_BLOCK_MCODE,        ///< Message code (bits [7:0] of byte 7) (default 0xff)
  PCIE_BLOCK_IO_ADDR,          ///< Address bits [31:0] in a 32-bit IO access
  PCIE_BLOCK_IO_RSVD111,       // Reserved (bits [1:0] of byte 11) (default 0x0)    
  PCIE_BLOCK_MSGAS_RSVD047,    // Reserved for MsgAS (bytes 4,5,6,7) (default 0x0)
  PCIE_BLOCK_MSGAS_RSVD087,    // Reserved for MsgAS (bytes 8,9,10,11) (default 0x0)
  PCIE_BLOCK_MSGAS_RSVD127,    // Reserved for MsgAS (bytes 12,13,14,15) (default 0x0)
  PCIE_BLOCK_MSG_BYTES08_11,    // Fields dependent on type of Message (bytes 8,9,10,11) (default 0x0)
  PCIE_BLOCK_MSG_BYTES12_15,    // Fields dependent on type of Message (bytes 12,13,14,15) (default 0x0)
  PCIE_BLOCK_MSGASD_RSVD047,    // Reserved for MsgASD (bytes 4,5,6,7) (default 0x0)
  PCIE_BLOCK_MSGASD_RSVD087,    // Reserved for MsgASD (bytes 8,9,10,11) (default 0x0)
  PCIE_BLOCK_MSGASD_RSVD127,    // Reserved for MsgASD (bytes 12,13,14,15) (default 0x0)
  PCIE_BLOCK_OBSOLETE_34,           
  PCIE_BLOCK_RESOURCE,          ///< Resource (\ref EPCIEBlockValResource ) (default PCIE_BLOCKV_RESOURCE_MEM)
  PCIE_BLOCK_REPEAT,
  ///< Number of times this line is repeated. 0 -- do not repeat, i.e. run
  ///only once(default 0x0) (max is 255)
  PCIE_BLOCK_PATTERN_TERM,      ///< Enables the pattern terms for this blockline. Bits 3-0 correspond to pattern terms 3-0 resp. The blockline execution is delayed/repeated as long as at least one of the enabled patterns terms is true. Default is 0x0, i.e. all four pattern terms are disabled. See PCIE_BLOCK_WAITREPEAT.
  PCIE_BLOCK_WAITREPEAT,        ///< Allows delaying or repeating the blockline until a certain condition is met. See \ref EPCIEBlockValWaitRepeat. Default is PCIE_BLOCKV_WAITREPEAT_REPEATPATTERN. If you additionally set PCIE_BLOCK_REPEAT, for instance to 4, it means you have set up 5 blocklines which all have PCIE_BLOCK_WAITREPEAT set (and each line has PCIE_BLOCK_REPEAT set to 0).
  PCIE_BLOCK_INTADDR,           ///< Internal Address in data memory (default 0x0)
  PCIE_BLOCK_MARK // This must be last ! (see pciehwcontrol.cpp)
} EPCIEBlock;

/// \brief Requester behavior properties
///
typedef DEV_MIDL_ENUM enum EPCIEReqBeh
{
  PCIE_REQBEH_EP,              ///< TLP is poisoned (default 0x0)  
  PCIE_REQBEH_TD,              ///< TLP digest present. This bit will only be shown in the header and has no effect (the format of the TLP will not change). (default 0x0)
  PCIE_REQBEH_INCORRECT_LCRC,  ///< Incorrect LCRC inserted (default 0x0)
  PCIE_REQBEH_NULLIFIED_TLP,   ///< TLP is nullified (LCRC is inverted and EDB is used as end symbol) (default 0x0)
  PCIE_REQBEH_WRONG_PYLD_SIZE, ///< Wrong payload size is inserted (default 0x0).
                               ///This field can be enabled or disabled. If enabled (0x1), the value in the LEN field of the packet is increased or increased by 1 (depending on \ref PCIE_EXERCISER_WRONG_PAYLOAD_SIZE_BEH which can be controlled by \ref ExerciserSet) before sending the packet.
                               ///ATTENTION: This feature is not supported for packets with payload size of 1024 DW.
  PCIE_REQBEH_INSERT_TD,       ///< The TLP Digest is inserted in the packet (default 0x0).
                               // PCIE_REQBEH_INSERT_TD replaces PCIE_REQBEH_WRONG_TD.
  PCIE_REQBEH_PRIORITY,        ///< Priority of this request compared to a request coming from another channel (default 0x0).
                               ///0x1 is low priority and 0xF is the highest priority. 
  PCIE_REQBEH_INCORRECT_ECRC,  ///< Incorrect ECRC is inserted (default 0x0)
  PCIE_REQBEH_INCORRECT_DISP,  ///< Incorrect Disparity is created (default 0x0)
  PCIE_REQBEH_AUTOTAG,         ///< Tag of the transaction is automatically created (default 0x1). For Gen3, tags are created automatically and this entry is invalid.
  PCIE_REQBEH_TAG,             ///< Tag of the transaction (if AUTOTAG is not
                               ///set) (default 0x0). For Gen3, this entry is invalid.
  PCIE_REQBEH_GAP,             ///< Gap inserted: gaps count between headers
                               ///of subsequent packets. Gap of current packet
                               ///will be taken until next packet is
                               ///started. GAP=1 means wait one DW time (== 4
                               ///symbols times) (default 0x0) (Max is 0xffff)
  PCIE_REQBEH_REPEAT,          ///< Number of times this line is repeated (0 means
                               /// execute once and do no repeat) (default 0x0) (Max is 0xff)
  PCIE_REQBEH_REPLACE_STP,     ///<1=Replace the Start TLP special symbol (K27.7) with an arbitray 9bit symbol, which can be set with property PCIE_DLLPHY_STP_SYMBOL (see function DllPhySet(), enum \ref EPCIEDllPhy). Default is 0.
  PCIE_REQBEH_REPLACE_END,     ///<1=Replace the END special symbol (K29.7) with an arbitray 9bit symbol, which can be set with property PCIE_DLLPHY_END_SYMBOL (see function DllPhySet(), enum \ref EPCIEDllPhy). Default is 0.
  PCIE_REQBEH_REPLACE_SEQNO,   ///<1=Replace packet's sequence number. Replaced sequence number is calculated by adding correct sequence number and the offset specified with property PCIE_DLLPHY_SEQNO_OFFSET (see function DllPhySet(), enum \ref EPCIEDllPhy). Default is 0.

  
  // Must be last entry !!!                             
  PCIE_REQBEH_LEN,             ///< Payload size in DWORDs (max. 1024 DWORDs) (default 0x1)
} EPCIEReqBeh;

/// \brief Completer block behavior properties
///
typedef DEV_MIDL_ENUM enum EPCIECompBeh
{
  PCIE_COMPBEH_EP,              ///< TLP is poisoned (default 0x0)  
  PCIE_COMPBEH_TD,              ///< TLP digest present. This bit will only be shown in the header and has no effect (TLP will not have an ECRC). (default 0x0)
  PCIE_COMPBEH_INCORRECT_LCRC,  ///< Incorrect LCRC inserted (default 0x0)
  PCIE_COMPBEH_NULLIFIED_TLP,   ///< TLP is nullified (LCRC is inverted and EDB is used as end symbol) (default 0x0)
  PCIE_COMPBEH_WRONG_PYLD_SIZE, ///< Wrong payload size is inserted (default 0x0). 
                                ///This field can be enabled or disabled. If enabled (0x1), the value in the LEN field of the packet is increased or increased by 1 (depending on \ref PCIE_EXERCISER_WRONG_PAYLOAD_SIZE_BEH which can be controlled by \ref ExerciserSet) before sending the packet.
                                ///ATTENTION: This feature is not supported for packets with payload size of 1024 DW.
  PCIE_COMPBEH_INSERT_TD,       ///< The TLP Digest is inserted in the packet (default 0x0).
                                // PCIE_COMPBEH_INSERT_TD replaces PCIE_COMPBEH_OBSOLETE_05.
  PCIE_COMPBEH_PRIORITY,        ///< Priority (default 0x0).0=Delay disabled,1=Delay Enabled
  PCIE_COMPBEH_INCORRECT_ECRC,  ///< Incorrect ECRC is inserted (default 0x0)
  PCIE_COMPBEH_INCORRECT_DISP,  ///< Incorrect Disparity is created (default 0x0)
  PCIE_COMPBEH_COMPSTATUS,      ///< Completion status (Spec 1.0a, Table 2-20): 0 = Successful Completion (default), 1 = Unsupported Request, 2 = Configuration Request Retry Status, 4 = Completer Abort
  PCIE_COMPBEH_RCB_COUNT,       ///< RCB count (range 0x0..0x3F, default 0x3F). 0 = Completion stops at next RCB. 1 = Completion crosses one RCB and stops at the second RCB 2 = Completion crosses two RCBs
  PCIE_COMPBEH_OBSOLETE_11,      
  PCIE_COMPBEH_OBSOLETE_12, 
  PCIE_COMPBEH_OBSOLETE_13,
  PCIE_COMPBEH_REPLACE_STP,     ///<1=Replace the Start TLP special symbol (K27.7) with an arbitrary symbol, which can be set with property PCIE_DLLPHY_STP_SYMBOL (see function DllPhySet(), enum \ref EPCIEDllPhy). Default is 0.
  PCIE_COMPBEH_REPLACE_END,     ///<1=Replace the END special symbol (K29.7) with an arbitrary symbol, which can be set with property PCIE_DLLPHY_END_SYMBOL (see function DllPhySet(), enum \ref EPCIEDllPhy). Default is 0.
  PCIE_COMPBEH_REPLACE_SEQNO,   ///<1=Replace packet's sequence number. Replaced sequence number is calculated by adding the correct sequence number and the offset specified with property PCIE_DLLPHY_SEQNO_OFFSET (see function DllPhySet(), enum \ref EPCIEDllPhy). Default is 0.
  PCIE_COMPBEH_DISCARD,         ///<1=Discard completion completely. This property is per packet and not per request (default 0x0).
  
  // Must be last entry !!!!!
  PCIE_COMPBEH_REPEAT          ///< Number of times this line is repeated (default 0x0) (max is 255)
} EPCIECompBeh;


/// \brief Link width settings
///
/// Using Tcl, you will not be able to use the enumeration. Please use the
/// number indicated in parentheses instead.
///
/// \note For the Exerciser, these values can be used as a bit field to allow
/// different link widths to be trained. To allow x8 and x4, set the
/// "PCIE_DLLPHY_LINKWIDTH_CAPABLE" to \code PCIE_LINKWIDTHV_X1 | PCIE_LINKWIDTHV_X8
/// \endcode
typedef DEV_MIDL_ENUM enum EPCIELinkWidthVal
{
  PCIE_LINKWIDTHV_X0 = 0, ///< (0) Link is down
  PCIE_LINKWIDTHV_X1 = 1, ///< (1) Link width is x1
  PCIE_LINKWIDTHV_X2 = 2, ///< (2) Link width is x2
  PCIE_LINKWIDTHV_X4 = 4, ///< (4) Link width is x4
  PCIE_LINKWIDTHV_X8 = 8,  ///< (8) Link width is x8
  PCIE_LINKWIDTHV_X16 = 16  ///< (16) Link width is x16
} EPCIELinkWidthVal;


// For compatibility only !!!
// \brief Spec Revision settings
//
// Using Tcl, you will not be able to use the enumeration. Please use the number indicated in parentheses instead.
typedef DEV_MIDL_ENUM enum EPCIESpecRevVal
{
  PCIE_SPECREV_1_0 = 0,       // (0) Hardware uses spec revision 1.0
  PCIE_SPECREV_1_0_a = 1      // (1) Hardware uses spec revision 1.0a 
} EPCIESpecRevVal;


/// \brief PCI Express Exerciser properties (*Changed in Gen3 Exerciser*)

///
typedef DEV_MIDL_ENUM enum EPCIEDllPhy
{
  PCIE_DLLPHY_LINKMODE_CAPABLE,
  ///< Capable link width (write bitmap with Bit0:x1, Bit1:x2, Bit2:x4,
  ///Bit3:x8, Bit4:x16, or see \ref EPCIELinkWidthVal. When training the link, the
  ///Exerciser negotiates the set link widths. \note You can
  ///disable the x1 mode (that is, enable x8 only) even though the specification makes
  ///x1 obligatory. Please be aware that all link widths are not supported for Gen3.
  ///\note Please ensure that you select only those linkwidths that your device supports. For instance,if
  /// your device is x1, you must always set this property to 1 to ensure proper linkup behavior.
  PCIE_DLLPHY_OBSOLETE_01,
  PCIE_DLLPHY_DISABLE_SCRAMBLE,
  ///< This will set the corresponding bit in Training Sequences to the specified value (1 or 0). 
  /// From now onwards, the 'Disable Scrambling' bit in the 'Training Control' field in all the sent TS will have the 
  /// specified value.
  PCIE_DLLPHY_OBSOLETE_03,
  PCIE_DLLPHY_FC_UPDATE_PERIOD,
  ///< The flow control update period represents the time interval in which
  ///all flow control update packets are sent. The unit is 2 Symbol
  ///FC Update Period value. Value Range = 1us to 511 us. Default = 25us  
  PCIE_DLLPHY_RESEND_PERIOD,
  ///< The resend period is used in the flow control initialization
  ///process. It controls the period when the flow
  ///control starts retransmitting FCInit packets. The unit used is Symbol Times. For details, see the PCIE
  ///Express Specification.
  PCIE_DLLPHY_TX_POLARITY_INVERSION_MAP,
  ///< Provides tx polarity inversion (bits [7:0] are lane [7:0], lane is
  ///inverted if bit is 1) (default is 0x0 which means no inversion)
  PCIE_DLLPHY_UPSTREAM_PORT_ENABLE,
  ///< Configures the behavior of the Exerciser during link initialization. If
  ///set to 1, the Exerciser acts as an upstream port, that is, as if it is connected to a
  ///downstream port of a switch or a root complex. If set to 0, the Exerciser
  ///acts as a downstream port and assigns the link number determined by \ref
  ///PCIE_DLLPHY_LINKNUMBER_ADVERTISED (default: 1 for "To Upstream"
  ///sessions and 0 for "To Downstream" sessions)
  PCIE_DLLPHY_SKIP_INTERVAL,
  ///< Determines the number of symbol times between two skip ordered
  ///sets. The value shall be between 1180 and 1538. For
  ///details, see the PCIE Express Specification. (default is 1280 and maximum value is 4095)
  PCIE_DLLPHY_LINKNUMBER_ADVERTISED,
  ///< Link Number advertised to link partner if operating as a downstream port.
  ///(see \ref PCIE_DLLPHY_UPSTREAM_PORT_ENABLE), (default 0)
  PCIE_DLLPHY_REPLAY_TIMER,
  ///< When the Exerciser receives no response on a request, the request is
  ///replayed after this time period. The unit used is 2 Symbol Times. (default is
  ///12500 Symbol Times)
  PCIE_DLLPHY_OUTPUT_ENABLE_DELAY,
  ///< When the PCI Reset is deasserted, wait this period of time until the
  ///outputs are enabled. Unit used is milli seconds. (default 250). This option is not valid for Gen3.
  PCIE_DLLPHY_DUT_CONNECTIVITY,
  ///< Enables handling of PCI Interrupts, depending on the use model. If the
  ///Exerciser is plugged into a system, use the Platform mode. This will cause the
  ///Exerciser to react on the PERST# signal. For cases where the exerciser
  ///should ignore PERST#, or it is not plugged into a system, set to the
  ///bench mode. Please refer to \ref EPCIEDllPhyDUTConnectivityVal. Default
  ///is 1 (PCIE_DLLPHY_DUT_CONNECTIVITYV_PLATFORM). This option is not valid for Gen3.
  PCIE_DLLPHY_ENABLE_LANE_REVERSAL,
  ///< If set to 1, lanes are reversed immediately.
  ///The reversed lanes are dependent on the capable link width (\ref PCIE_DLLPHY_LINKMODE_CAPABLE). 
  ///For example, if the capable link width is 4, the lane order is changed from 0,1,2,3 to 3,2,1,0.\n
  ///Please be aware of the property \ref PCIE_DLLPHY_ENABLE_DETECT_LANE_REVERSAL that is used to control the 
  ///lane reversal capabilites of the Exerciser.
  PCIE_DLLPHY_STP_SYMBOL,
  ///<Replaces the Start TLP special symbol (K27.7) with an arbitrary symbol.
  ///Bit 8 denotes the control bit (0=Data,1=Special symbol).
  ///Default is 0x1FB (i.e. Start TLP special symbol K27.7).
  ///To enable this, you additionally need to set the corresponding property
  ///PCIE_REQBEH_REPLACE_STP, PCIE_COMPBEH_REPLACE_STP and/or PCIE_SI_BUSERROR.
  PCIE_DLLPHY_END_SYMBOL,
  ///<Replaces the END special symbol (K29.7) with an arbitrary symbol.
  ///Bit 8 denotes the control bit (0=Data,1=Special symbol).
  ///Default is 0x1FD (i.e. END special symbol K29.7).
  ///To enable this, you additionally need to set the corresponding property
  ///PCIE_REQBEH_REPLACE_END, PCIE_COMPBEH_REPLACE_END and/or PCIE_SI_BUSERROR.
  PCIE_DLLPHY_SEQNO_OFFSET,
  ///<Replaces the sequence number of packets to test the NAK engine.
  ///Sequence number will be wrong once and correct on replay.
  ///To enable this, you additionally need to set the corresponding property 
  ///PCIE_REQBEH_REPLACE_SEQNO, PCIE_COMPBEH_REPLACE_SEQNO and/or PCIE_SI_BUSERROR.
  ///Replaced sequence number is calculated by adding correct sequence number and this offset.
  PCIE_DLLPHY_DELAY_COMP_QUEUE0,
  ///<Programmable delay of completions in queue 0.
  ///The completion is only delayed, if it additionally has priority 0 (from completion behavior, see property PCIE_COMPBEH_PRIORITY).
  ///The delay is given as a multiple of 16 ns. Minimum is 0 (default) and maximum is 0xFFFFFF (i.e. about 268ms)
  PCIE_DLLPHY_DELAY_COMP_QUEUE1,
  ///<Programmable delay of completions in queue 1.
  ///The completion is only delayed, if it additionally has priority 0 (from completion behavior, see property PCIE_COMPBEH_PRIORITY).
  ///The delay is given as a multiple of 16 ns. Minimum is 0 (default) and maximum is 0xFFFFFF (i.e. about 268ms)
  PCIE_DLLPHY_DELAY_COMP_QUEUE2,
  ///<Programmable delay of completions in queue 2.
  ///The completion is only delayed, if it additionally has priority 0 (from completion behavior, see property PCIE_COMPBEH_PRIORITY).
  ///The delay is given as a multiple of 16 ns. Minimum is 0 (default) and maximum is 0xFFFFFF (i.e. about 268ms)
  PCIE_DLLPHY_DELAY_COMP_QUEUE3,
  ///<Programmable delay of completions in queue 3.
  ///The completion is only delayed, if it additionally has priority 0 (from completion behavior, see property PCIE_COMPBEH_PRIORITY).
  ///The delay is given as a multiple of 16 ns. Minimum is 0 (default) and maximum is 0xFFFFFF (i.e. about 268ms)
  PCIE_DLLPHY_CROSSLINK_CAPABLE,
  ///<This is a capability feature to enable a downstream port to connect with another downstream port and form a link 
  ///or upstream port to connect with another upstream port and form a link. If this is enabled, a downstream port can still connect 
  ///to an upstream port to form a link.
  PCIE_DLLPHY_MAX_FTS_GEN1,
  ///<Use this to modify the N_FTS field shown in the Training Sequences that the Exerciser sends in Gen1 speed.  
  ///(default is 255 and maximum is 255).  Attention: It is not advisable to change the default setting of FTS number as the behaviour may 
  ///be undefined if the setting are changed.
  PCIE_DLLPHY_MAX_FTS_GEN2,
  ///<Use this to modify the N_FTS field shown in the Training Sequences that the Exerciser sends in Gen2 speed. 
  ///(default is 254 and maximum is 255). Attention: It is not advisable to change the default setting of FTS number as the behaviour may 
  ///be undefined if the setting are changed.
  PCIE_DLLPHY_LINKSTATEDIRECT_TIMEOUT,
  ///<When using the direct state feature (\ref LinkStateDirect), use this to specify how long the LTSSM stays 
  ///in the directed states 'Loopback', 'Hot Reset', 'Disabled' and 'L0s'. The time period is in terms of clock 
  ///cycles of 16ns. The range is from 0x1-0xFFFFFF.
  PCIE_DLLPHY_TS_FORCE_PAD,
  ///<Within the Training Sequences TS1 and TS1, the Symbol 'PAD' (K.23.7) is forcibly inserted. 
  ///The position within the TS is Symbol Number 1 and 2 which contain 'Link Number within component' and 
  ///'Lane Number with Port' respectively. The value is a bitmap where the least significant bit refers to Lane 0.  
  PCIE_DLLPHY_L0S_ENTRY_TIMEOUT,
  ///<This specifies how long the link has to be in the idle state which means that no DLLP/TLP are being transmitted 
  ///until the LTSSM enters L0s on the Tx side. The unit is 16ns. The range is from 0x1-0xFFFFFF.
  PCIE_DLLPHY_ENABLE_DETECT_LANE_REVERSAL,
  ///<This is a switch to enable the Exerciser to handle the lane reversal feature if the lanes are reversed between 
  ///DUT and Exerciser. If you want to reverse lanes, use \ref PCIE_DLLPHY_REVERSE_LANES.
  ///\li A) If the DUT initiates link training and the lane reversal switch of Exerciser is enabled, the 
  ///Exerciser will handle lane reversal on TX and RX sides.
  ///\li B) If the DUT initiates link training and the lane reversal switch of Exerciser is not enabled, 
  ///the Exerciser will not handle lane reversal on TX and RX sides. The DUT will receive reversed lane numbers and 
  /// will have to handle lane reversal in that case. This switch is used to test DUT's lane reversal feature 
  ///when the DUT is a Downstream Port/Root complex.
  PCIE_DLLPHY_SKIP_SYMBOLS_NUM,
  ///<This specifies the number of SKP symbols in SKP Ordered set. (default is 0x3, maximum is 5, and minimum is 1)
  PCIE_DLLPHY_FC_TIMEOUT,
  ///<If the Exerciser does not receive a FC Update packet for this period of time, it is considered a Flow Control update timeout. To 
  ///check whether a timeout occured, use the protocol error feature (\ref ProtocolErrorFirstRead).
  ///FC timeout value [unit = 1us]. Value Range = 1us to 511 us. Default = 30us
  PCIE_DLLPHY_DEEMPHASIS_LEVEL_GEN1,
  ///< This value controls the de-emphasis settings at Gen 1 (2.5 GT/s) speed. A value of 0x0 indicates Disabled, 0x1 indicates -3.5 dB de-emphasis. PCIE_DLLPHY_DEEMPHASIS_LEVEL_GEN1 replaces PCIE_DLLPHY_DEEMPHASIS_LEVEL.
  ///(default is 0x1: -3.5 dB )
  PCIE_DLLPHY_DEEMPHASIS_LEVEL_GEN2,
  ///< This value controls the de-emphasis settings at Gen 2 (5.0 GT/s) speed. A value of 0x0 indicates Disabled, 0x1 indicates -3.5 dB de-emphasis and 0x2 indicates -6 dB de-emphasis. PCIE_DLLPHY_DEEMPHASIS_LEVEL_GEN2 replaces PCIE_DLLPHY_DEEMPHASIS_ENABLE.
  ///(default is 0x1: -3.5 dB )
  PCIE_DLLPHY_DEEMPHASIS_IN_TS,
  ///< This value controls the de-emphasis settings in TS ordered sets. A value of 1b indicates -3.5 dB de-emphasis and a value of 0b
  ///indicates -6 dB de-emphasis. Please see \ref PCIE_DLLPHY_DEEMPHASIS_ENABLE and \ref PCIE_DLLPHY_DEEMPHASIS_LEVEL.
  ///(default is 0x1)
  PCIE_DLLPHY_FC_UPDATE_PERIOD_ENABLE,
  ///< This specifies if FC update packets are being sent each time the timeout counter \ref PCIE_DLLPHY_FC_UPDATE_PERIOD
  ///expires. ('1'= enabled, '0' = disabled)
  ///(default is 0x1)
  PCIE_DLLPHY_L0S_TIMEOUT_ENABLE,
  ///<The specifies if the LTSSM state 'L0s' is exited into the 'Recovery' state after a timer period specified in \ref PCIE_DLLPHY_LINKSTATEDIRECT_TIMEOUT.
  ///This behavior can be enabled ('1') or disabled ('0'). 
  ///(default is 0x1)
  PCIE_DLLPHY_L0S_EXIT_SKP_OS_ENABLE,
  ///<This specifies whether SKP Ordered Sets are sent on exit from the LTSSM State 'L0s' into the state 'Recovery'.
  ///('1' = 'Send Skip Ordered Sets' and '0' = 'Do not send SKP Ordered Sets)
  ///(default is 0x1)
  PCIE_DLLPHY_L0S_EXIT_FTS_NUM,
  ///<This specifies the number of FTS transmitted on exit from the LTSSM State 'L0s' into the state 'Recovery'.
  ///The feature can be enabled using \ref PCIE_DLLPHY_L0S_EXIT_FTS_NUM_ENABLE.
  ///The range is from 0x1 to 0x3FFFF.
  PCIE_DLLPHY_L0S_EXIT_FTS_NUM_ENABLE,
  ///<The number of FTS transmitted on exit from the LTSSM State 'L0s' into the state 'Recovery'
  ///is equal to the number of received TS Ordered Sets. To modify the number of FTS that are
  ///transmitted in this case, set this property to '1' and spcecify the number of FTS via
  ///\ref PCIE_DLLPHY_L0S_EXIT_FTS_NUM.
  ///(default is 0x0)
  PCIE_DLLPHY_ASPM_L1_ENTRY_TIMEOUT,
  ///<This specifies how long the link has to be in the idle state (meaning: no DLLP/TLP being transmitted) 
  ///until the LTSSM enters the L1 state. The unit used is 16ns.
  PCIE_DLLPHY_L1_EXIT_TIMEOUT,
  ///<The time period after which the exit from L1 is initiated by a downstream component. One unit is 16ns. 
  ///This value is not useful for an upstream component.
  PCIE_DLLPHY_L1_EXIT_TIMEOUT_ENABLE,
  ///<This specifies whether LTSSM state 'L1' is exited into the state 'Recovery' after a timer period specified in \ref PCIE_DLLPHY_L1_EXIT_TIMEOUT.
  ///This behavior can be enabled ('1') or disabled ('0'). 
  ///(default is 0x1)
  PCIE_DLLPHY_PCIPM_L1_AGRESSIVE,
  ///<This enables agressive (1) L1 Power Management or passive (0) L1 Power Management. 
  ///When the device is in L1 and receives a packet, it will respond to the packet and then go back into L1 if
  ///agressive is enabled. If passive is enabled, the device will not go back to L1 after responding to the packet.
  ///This is only valid when the Exerciser acts as a downstream device.
  PCIE_DLLPHY_CROSSLINK_TIMEOUT,
  ///<This is the timeout value for the feature \ref PCIE_DLLPHY_CROSSLINK_CAPABLE. The unit is 16ns. The range
  ///is from 0x1 to 0xffffff.
  PCIE_DLLPHY_LINK_UPCONFIGURE_IN_TS,
  ///<This will set the 'Link Upconfigure Capability' bit in the Training Sequence (Symbol 4, Bit6) to the value specified (default is 0x0). This is not valid for Gen3 (Release 1).
  PCIE_DLLPHY_MGT_PROTECTION_ENABLE,
  ///<This will enable ('1') or disable ('0') the MGT Protection. (default is 0x1)
  PCIE_DLLPHY_TOPCONNECTOR_ENABLE,
  ///<This specifies wether all signals are routed to top connector (1) or bottom connector (0). Default is 0.
  PCIE_DLLPHY_COMPLETION_TIMEOUT,
  ///<If the Exerciser does not receive a completion for this period of time, it is considered a completion timeout. To 
  ///check whether a timeout occured, use the protocol error feature (i.e. \ref ProtocolErrorFirstRead). The
  ///unit of the value is (50us). (maximum is 0x7ff and default is 0xc8 which is about 10ms)
  PCIE_DLLPHY_EIDLE_SENSOR_ENABLE,
  ///<This enables Electrical Idle Sensing on Receiver side. When set, the Electrical Idle on
  ///the Receiver side is detected when proper differential input signal is received. (default is 0 which means Electrical Idle Sensor turned off on receivers differential input signal) 
  PCIE_DLLPHY_PERFORMANCE_MODE,
  ///<To reduce the latencies for TLP Updates and ACK, set the value to '1' which means the 'Performance Mode' is enabled. 
  ///The latency reduction is done by bypassing a component (ECB), which might introduce Power Management related issues.
  ///The performance mode can be disabled by setting the value to '0'. This is also refered to as 'Normal Mode'. The default value is '0'.
  PCIE_DLLPHY_TX_PRESET_VALUE,
  ///<This specifies the Tx Preset value for the Exerciser SERDES. The range is from 0-15.
  PCIE_DLLPHY_REQUEST_PEER_DEVICE_TO_USE_PRESETS_ENABLE,
  ///<A value of 1b indicates that the Exerciser requests peer device to use Presets in Recovery
  ///Equalization sub-state. A value of 0b indicates that the Exerciser does not request peer device to use Presets.
  ///For Preset values to be requested, see PCIE_DLLPHY_TX_PRESET_VALUE_REQUEST_TO_PEER_DEVICE.
  PCIE_DLLPHY_TX_PRESET_VALUE_REQUEST_TO_PEER_DEVICE,
  ///<This specifies the Tx Preset value requested in TS OS to the peer device. The range is from 0-15.
  PCIE_DLLPHY_REQUEST_PEER_DEVICE_TO_USE_COEFFICIENTS_ENABLE,
  ///<A value of 1b indicates that the Exerciser would request peer device to use Coefficients in Recovery 
  ///Equalization sub-state and value of 0b indicates that the Exerciser would not request peer device to 
  ///use Coefficients. For Coefficient values to be requested, 
  ///see PCIE_DLLPHY_TX_PRE_CURSOR_COEFFICIENTS_REQUEST_TO_PEER_DEVICE.
  PCIE_DLLPHY_TX_PRE_CURSOR_COEFFICIENTS_REQUEST_TO_PEER_DEVICE,
  ///<This specifies the Tx Pre Cursor Coefficient value requested in TS OS to the peer device. 
  ///The range is from 0-63. 
  PCIE_DLLPHY_TX_CURSOR_COEFFICIENTS_REQUEST_TO_PEER_DEVICE,
  ///<This specifies the Tx Cursor Coefficient value requested in TS OS to the peer device. 
  ///The range is from 0-63.
  PCIE_DLLPHY_TX_POST_CURSOR_COEFFICIENTS_REQUEST_TO_PEER_DEVICE,
  ///<This specifies the Tx Post Cursor Coefficient value requested in TS OS to the peer device. 
  ///The range is from 0-63.
  PCIE_DLLPHY_RX_PRESET_HINTS_REQUEST_TO_PEER_DEVICE,
  ///<This specifies the Rx Preset Hint value requested in TS OS to the peer device. 
  ///The range is from 0-7.
  PCIE_DLLPHY_QUIESCE_GUARANTEE,
  ///<A value of 1b indicates that the Quiesce Guarantee bit will be set in TS OS while making request to redo 
  ///the Equalization in Recovery state. A value of 0b indicates that the Quiesce Guarantee bit will be 
  ///cleared in TS OS while making request to redo the Equalization in Recovery state.
  PCIE_DLLPHY_PERFORM_EQUALIZATION_IN_RECOVERY_AT_GEN3_ENABLE,
  ///<If Exerciser is USC, then the value of 1b indicates that the Exerciser would perform Equalization while going to    
  ///8 GT/s or in 8 GT/s in Recovery state. A value of 0b indicates that the Exerciser would not perform Equalization.
  PCIE_DLLPHY_REQUEST_FOR_EQUALIZATION_IN_RECOVERY_AT_GEN3_ENABLE,
  ///<If Exerciser is DSC, then the value of 1b indicates that the Exerciser would request for Equalization in Recovery 
  ///state at 8GT/s. The value of 0b indicates that the Exerciser would not request for Equalization in Recovery at 8GT/s.
  PCIE_DLLPHY_AUTONOMOUS_EQUALIZATION_ENABLE,
  ///<Autonomous Equalization Enable
  /// \li 1 -> If the Exerciser and peer device is 8GT/s capable and Exerciser is USC, then Exerciser would automatically initiate speed change to 8GT/s just after coming to L0 from Detect before doing DL_ACTIVE.
  /// \li 0 -> Exerciser would come to L0 at 2.5GT/s after Detect and do normal DL_ACTIVE after there. It would not automatically change speed to 8GT/s.
  PCIE_DLLPHY_PERFORM_PHASE2_PHASE3_EQUALIZATION_ENABLE,
  ///<Perform_phase2_phase3_equalization: Applicable only if Exerciser is USC and the "Perform Equalization" bit is set.
  /// \li 1 -> If Exerciser performs equalization at 8GT/s, it will not skip phase2 and phase3 of equalization.
  /// \li 0 -> Exerciser will skip phase2 and phase3 of equalization and go to recover_rcvrlock after performing phase1 of equalization.
	PCIE_DLLPHY_GXB_EQ_LOAD_CMD,
	PCIE_DLLPHY_GXB_RX_EQ_DC_GAIN,
  ///<This specifies the Rx Equalization DC Gain values for GXBs. 
  /// \li 000b -> 0 dB
  /// \li 001b -> 3 dB
  /// \li 010b -> 6 dB
  /// \li 011b -> 9 dB
  /// \li 100b -> 12 dB
	PCIE_DLLPHY_GXB_TX_OUT_VOLTAGE,
  ///<This specifies the Tx Output Voltage Control. 
  /// \li 000b -> 200mv
  /// \li 001b -> 400mv
  /// \li 010b -> 600mv
  /// \li 011b -> 700mv
  /// \li 100b -> 800mv
  /// \li 101b -> 900mv
  /// \li 110b -> 1000mv
  /// \li 111b -> 1200mv
	PCIE_DLLPHY_GXB_TX_PREEMPHESIS_PRE_TAP,
  ///<This specifies the Tx Preemphesis Pre Tap value for the Exerciser. 
  ///The range is from 0-31.
	PCIE_DLLPHY_GXB_TX_PREEMPHESIS_FIRST_POST_TAP,
  ///<This specifies the Tx Preemphesis First Post Tap value for the Exerciser. 
  ///The range is from 0-31.
	PCIE_DLLPHY_GXB_TX_PREEMPHESIS_SECOND_POST_TAP,
  ///<This specifies the Tx Preemphesis Second Post Tap value for the Exerciser. 
  ///The range is from 0-31.
	PCIE_DLLPHY_CHIP_RX_EQ_AC_GAIN,
  ///<This specifies the Rx Equilizer Chip AC gain
  ///The range is from 0-16.
  PCIE_DLLPHY_TX_PRE_CURSOR_COEFFICIENTS,
  ///<This specifies the Tx Pre Cursor Coefficient value for the Exerciser. 
  ///The range is from 0-63. 
  PCIE_DLLPHY_TX_CURSOR_COEFFICIENTS,
  ///<This specifies the Tx Cursor Coefficient value for the Exerciser. 
  ///The range is from 0-63.
  PCIE_DLLPHY_TX_POST_CURSOR_COEFFICIENTS,
  ///<This specifies the Tx Post Cursor Coefficient value for the GXbx Exerciser. 
  ///The range is from 0-63.
  PCIE_DLLPHY_TX_FS,
  ///<This specifies the Tx FS value for the Exerciser. 
  ///The range is from 12-63.
  PCIE_DLLPHY_TX_LF,
  ///<This specifies the Tx LF value for the Exerciser. 
  ///The range is from 0-63.
  PCIE_DLLPHY_TX_IGNORE_COEFF_RULE1_C1_FS,
  ///<This will enable (1) or disable (0) the Ignore Coefficient Rule check |C-1| <= Floor (FS/4).
  PCIE_DLLPHY_TX_IGNORE_COEFF_RULE2_C01_FS,
  ///<This will enable (1) or disable (0) the Ignore Coefficient Rule check |C-1|+C0+|C+1| = FS.
  PCIE_DLLPHY_TX_IGNORE_COEFF_RULE3_C01_LF,
  ///<This will enable (1) or disable (0) the Ignore Coefficient Rule check C0 -|C-1|-|C+1 |>= LF.
  PCIE_DLLPHY_SKIP_INTERVAL_GEN3,
  ///< Determines the number of blocks times between two skip ordered
  ///sets. (the value should be between 370 and 375. For
  ///details, see the PCIE Express Specification).
  PCIE_DLLPHY_SKIP_SYMBOLS_NUM_GEN3,
  ///<This specifies the number of Symbols in SKP Ordered set. Valid values are 0-4.  
  ///(0:8 symobols, 1:12 symobols , 2:16 symobols , 3:20 symobols , 4:24 symobols)  
  ///For details, see the PCIE Express Specification.
  PCIE_DLLPHY_RX_LOAD_LFSR_BY_SKIP_ENABLE,
  ///<This will enable ('1') or disable ('0') the loading of descrambler LFSR by incoming SKIP OS.
  PCIE_DLLPHY_EXERCISER_MODE,
  ///<This specifies the Exerciser Protocol Mode. 
  /// \li 00b -> Only PCIe Ready
  /// \li 01b -> SRIOV Ready
  /// \li 10b -> MRIOV Ready
  PCIE_DLLPHY_TAG_MODE,
  ///<This specifies the Exerciser Tag Mode. 
  /// \li 0 -> Normal Tag Mode
  /// \li 1 -> Extended Tag Mode
  PCIE_DLLPHY_PCIE_SPEC_REVISION,
  ///<PCIe Specification number.
  /// \li "0.7"
  /// \li "0.7 with 0.9 TS"
  PCIE_DLLPHY_MAX_FTS_GEN3,
  ///<Use this to modify the N_FTS field shown in the Training Sequences that the Exerciser sends in Gen3 speed. 
  ///(default is 254 and maximum is 255). Attention: It is not advisable to change the default setting of FTS number as the behaviour may 
  ///be undefined if the setting are changed.
  PCIE_DLLPHY_DELAY_COMP_QUEUE4,
  ///<Programmable delay of completions in queue 4.
  ///The completion is only delayed, if it additionally has priority 0 (from completion behavior, see property PCIE_COMPBEH_PRIORITY).
  ///The delay is given as a multiple of 16 ns. Minimum is 0 (default) and maximum is 0xFFFFFF (i.e. about 268ms)
  PCIE_DLLPHY_REQUEST_EQUALIZATION_PRESET_IN_PHASE2_PHASE3,
  ///<Request Equalization Preset in Phase 2/Phase 3.
  ///If this is set, Exerciser would request Preset instead of coefficients in Phase2/Phase3 of Equalization.
  /// \li 0 -> Disabled (Default)
  /// \li 1 -> Enabled
  PCIE_DLLPHY_SEND_INVALID_SYNC_BITS,
  ///<Send invalid Sync bits till next recovery
  PCIE_DLLPHY_REQUEST_FOR_EQUALIZATION_IN_RECOVERY_AT_GEN3_ENABLE_PULSE,
  ///<If Exerciser is DSC, then setting the value to 1b will make the Exerciser request for Equalization in Recovery 
  ///state at 8GT/s. If this bit is to 0b, then the Exerciser would not request for Equalization in Recovery at 8GT/s. 
  /// This bit will get cleared automatically when the Exerciser goes into recovery.
  /// \li 1 -> Enable
  /// \li 0 -> Disable
  PCIE_DLLPHY_REJECT_PRESET_COEFF_N_TIMES_VALUE,
  ///<Reject Preset Coeff N Times Value
  ///The range is from 0-63. 
  PCIE_DLLPHY_LTSSM_FLAG0,
  ///<This provides manual control over LTSSM flag.User can set or clear the flag using this
  /// \li 1 -> Set LTSSM Flag0
  /// \li 0 -> Clear LTSSM Flag0
  PCIE_DLLPHY_LTSSM_FLAG1,
  ///<This provides manual control over LTSSM flag.User can set or clear the flag using this
  /// \li 1 -> Set LTSSM Flag1
  /// \li 0 -> Clear LTSSM Flag1
  PCIE_DLLPHY_EQ_RX_AC_GAIN_USING_RX_PRESET_HINTS_ENABLE,
  ///<This will enable ('1') or disable ('0') the use of Rx Preset Received to program Rx AC gain.
  PCIE_DLLPHY_EQ_ENABLE_GEN3_AUTOMATIC_TRANSMITTER_DEEMPHESIS,
  ///<This will enable ('1') or disable ('0') Automatic transmitter deemphesis
  PCIE_DLLPHY_EQ_TX_LOAD_COEFFICIENT_VALUES,
  ///<This will Load Coefficient Values.
  PCIE_DLLPHY_EQ_ENABLE_GEN3_AUTOMATIC_COEFFICIENT_REQUEST,
  ///<This will enable ('1') or disable ('0') the Automatic Coefficient Request.
  PCIE_DLLPHY_INITIATE_LINK_RECOVERY_ON_NO_UFC  
  ///<If enable ('1') the Exerciser would initiate recovery if no UpdateFC has been seen at receiver for 200us in L0, L0s state.
} EPCIEDllPhy;


// There is a copy of this enum in the analyzer ! TBD move stuff in control interface
///
/// Values for PCIE_DLLPHY_DUT_CONNECTIVITY,
/// which handles the way the Exerciser behaves upon assertion or deassertion of the sideband signal PERST#
typedef DEV_MIDL_ENUM enum EPCIEDllPhyDUTConnectivityVal
{
  PCIE_DLLPHY_DUT_CONNECTIVITYV_BENCH = 0,   ///< (0) The probeboard is not inserted into a platform. The Exerciser will ignore PERST#. You can use IAgtPCIEControl::ProbeBoardExecute to control the PERST# line for the add-in card plugged into the probe board.
  PCIE_DLLPHY_DUT_CONNECTIVITYV_PLATFORM = 1 ///< (1) The probeboard is inserted into a platform. The exerciser will respect the PERST# signal; outputs to system or add-in-card will be shut off while PERST# is asserted. 
} EPCIEDllPhyDUTConnectivityVal;


/// \brief Exerciser for PCI Express properties
///
/// IAgtPCIEExerciser::ExerciserSet, IAgtPCIEExerciser::ExerciserGet
typedef DEV_MIDL_ENUM enum EPCIEExerciser
{
  PCIE_EXERCISER_REQUESTER_ID,
  ///< Requester ID (bits[15:8] Bus Number, bits[7:3] Device Number, bits[2:0]
  ///Function Number) (default 0x0). These will be used in all requests
  ///generated by the Exerciser. ATTENTION: This value is updated by hardware 
  ///everytime a Config Type 0 Write Request is received.
  PCIE_EXERCISER_DATACMP_ENABLE,
  ///< This call is invalid for Gen3. For Gen3, enable the data comparison per decoder (see \ref EPCIEDecValResource and \ref DecWrite)
  ///and for generic enable for all completions (\ref PCIE_EXERCISER_DATACMP_COMPLETIONS_ENABLE).
  ///Enables or disables the data compare mode. When enabled, the internal data memory is used
  ///for reference data and the incoming data (either through write requests or
  ///read completions) is compared against the reference data. Errors are
  ///reported in Exerciser Status (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode. (Default is 0 which means data compare mode is disabled)
  PCIE_EXERCISER_PERF_RX_PATTERN,
  ///< Determines the enabled pattern terms for RX statistics. If Bit0 = 1, it enables pattern term 0. If Bit1 = 1, it enables pattern term 1 etc.
  /// Default is 0.
  /// See also PCIE_PERFORMANCECOUNTERSTATUS_RX_PATTERN. This is not available for E2960B.
  PCIE_EXERCISER_WRONG_PAYLOAD_SIZE_BEH,
  ///< This controls how the value in the LEN field of the packet is modified when sending a packet with 
  ///wrong payload size via Send Immediate (\ref PCIE_SIV_BUSERROR_WRONG_PAYLOAD_SIZE), the Requester Behaviour for the Block Memories (\ref PCIE_REQBEH_WRONG_PYLD_SIZE) 
  ///or the completer behaviour (\ref PCIE_COMPBEH_WRONG_PYLD_SIZE).
  ///Possible values are:
  /// \li (0) The payload size in the LEN field is the correct payload size incremented by one (default). 
  /// \li (1) The payload size in the LEN field is the correct payload size decremented by one.
  PCIE_EXERCISER_DATA_RATE,
  ///< This is used to specify the data rate capability of the Exerciser. The value is reflected in the config space
  ///capability 'PCI Express Capabilities' in the register 'Link Capabilities' at offset 0xc. Please be aware that 
  ///to modify the link speed, it is neccessary to set this value to 'generation 2' speed as well as performing a
  ///call to \ref LinkStateDirect with the property \ref PCIE_LINKSTATEDIRECT_SPEED_CHANGE.
  ///Value = 1, generation 1 (2.5 Gb/s) data rate supported\n
  ///Value = 2, generation 2 (5.0 Gb/s) data rate as well as generation 1 data rate supported\n
  ///Value = 4, generation 3 (8.0 Gb/s) data rate as well as generation 1 and generation 2 data rates supported\n
  PCIE_EXERCISER_FPGA_CONFIGURATION,
  ///< This returns the current FPGA configuration. For details, refer to \ref EPCIEFPGAConfiguration.
  PCIE_EXERCISER_PERFORMANCECOUNTER_DLLPHY_ERR_LANE,
  ///< This specifies which lane is used with the performance counters PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_CODING_ERR and PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_DISPARITY_ERR
  ///Set this value first, then start with the performance measurement. This enum entry is only valid for E2960B and has no effect on E2960A.
  PCIE_EXERCISER_DATACMP_HWCHANNEL_A_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function A. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare).
  ///PCIE_EXERCISER_DATACMP_COMPLETIONS_ENABLE is deprecated. Data compare can now be used on a per function basis. (*New in Gen3 Exerciser*)
  PCIE_EXERCISER_VC_RESOURCE_1_ENABLE,
  ///< Enables the VC resource 1 of the Exerciser. Valid only in the PCIE/SRIOV mode. (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_MRIOV_PER_VH_FLOW_CONTROL,
  ///< Enables per VH flow control. (*New in Gen3 Exerciser*). 
  PCIE_EXERCISER_MRIOV_VH1_ENABLE,
  ///< Enables the Virtual hierarchy VH1. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*). 
  PCIE_EXERCISER_MRIOV_VH2_ENABLE,
  ///< Enables the Virtual hierarchy VH2. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*). 
  PCIE_EXERCISER_MRIOV_VH3_ENABLE,
  ///< Enables the Virtual hierarchy VH3. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*). 
  PCIE_EXERCISER_ISP_ACCESS_MODE,
  ///< Enable PCIe Port/ISP Access.(1=Enable,0=Disable) (*New in Gen3 Exerciser*). 
  PCIE_EXERCISER_ARI_CAPABILITY_ENABLE,
  ///< Enables ARI capability. (*New in Gen3 Exerciser*). 
  PCIE_EXERCISER_MRIOV_VH4_ENABLE,
  ///< Enables the Virtual hierarchy VH4. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_B_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function B. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_C_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function C. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_D_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function D. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_E_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function E. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_BVF1_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function B VF1. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_BVF2_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function B VF2. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_CVF1_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function C VF1. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_CVF2_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function C VF2. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_DVF1_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function D VF1. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_DVF2_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function D VF2. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_EVF1_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function E VF1. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_DATACMP_HWCHANNEL_EVF2_COMPLETIONS_ENABLE,
  ///< This enables data compare mode for Function E VF2. When enabled, the internal data memory is used
  ///for reference data and the incoming data through read completions is compared against the reference data. Errors are
  ///reported in ExerciserStatus (see #ExerciserStatusRead, and \ref
  ///EPCIEExerciserStatus). No data is written to the data memory in this
  ///mode (default is 0 which means no data compare). (*New in Gen3 Exerciser*).
  PCIE_EXERCISER_PATTERN_TERM_0_ENABLE,
  ///< This enables pattern matching for pattern term 0. This has similar functionality as PatternRun, but this only enabled one pattern term while PatternRun enables all pattern terms.
  PCIE_EXERCISER_PATTERN_TERM_1_ENABLE,
  ///< This enables pattern matching for pattern term 1. This has similar functionality as PatternRun, but this only enabled one pattern term while PatternRun enables all pattern terms.
  PCIE_EXERCISER_PATTERN_TERM_2_ENABLE,
  ///< This enables pattern matching for pattern term 2. This has similar functionality as PatternRun, but this only enabled one pattern term while PatternRun enables all pattern terms.
  PCIE_EXERCISER_PATTERN_TERM_3_ENABLE,
  ///< This enables pattern matching for pattern term 3. This has similar functionality as PatternRun, but this only enabled one pattern term while PatternRun enables all pattern terms.
  PCIE_EXERCISER_APPLICATION_MODE,
  ///< Indicates current application mode. This has two possible values:
  /// \li 0x0: General (Default). This mode indicates the default PCIe Exerciser mode. NVMe functionality will be disabled in this mode.
  /// \li 0x1: NVMe mode. NVMe will be enabled in this mode. SRIOV and MRIOV will not be available in this mode.
  PCIE_EXERCISER_STORAGE_INTERRUPT_OPTION,
  ///< Indicates the interrupt option to be used. This has 3 possible values:
  /// \li 0x0: None
  /// \li 0x1: INTx
  /// \li 0x2: MSI
  /// \li 0x3: MSI-X
} EPCIEExerciser;
 
/// \brief PCIE Exerciser FPGA Configurations
///
typedef DEV_MIDL_ENUM enum EPCIEFPGAConfiguration
{
  PCIE_FPGACONFIGURATION_UNKNOWN = 0,///< (0) This FPGA Configuration is unknown.
  PCIE_FPGACONFIGURATION_COSIM = 1, // This is not documented on purpose.
  PCIE_FPGACONFIGURATION_COSIM_SECOND = 2, // This is not documented on pupose.
  PCIE_FPGACONFIGURATION_LTSSM_ONLY = 3,///< (3) This is LTSSM Tester only. The Exerciser is not included.
  PCIE_FPGACONFIGURATION_EXERCISER_ONLY = 4,///< (4) This is the Exerciser only. The LTSSM Tester is not included.
  PCIE_FPGACONFIGURATION_ALL = 5,///< (5) This includes the Exerciser as well as the LTSSM Tester.
  PCIE_FPGACONFIGURATION_PTC = 6,///< (6) This is the PTC image.
} EPCIEFPGAConfiguration;


/// \brief PCIE Exerciser properties for Block Memory
///
/// IAgtPCIEExerciser::BlockGenSet, IAgtPCIEExerciser::BlockGenGet, IAgtPCIEExerciser::BlockGenDefaultSet
typedef DEV_MIDL_ENUM enum EPCIEBlockGen
{
  PCIE_BLOCKGEN_STARTLINE,///< Startline in Block Memory (default 0x0). You can use it to quickly change between tests. You can pre-program several tests in subsequent areas of the Block Memory. Then use this property to determine where a test begins and PCIE_BLOCKGEN_STOPLINE to determine where it stops. The maximum value is the line specified in \ref PCIE_BLOCKGEN_STOPLINE.
  PCIE_BLOCKGEN_STOPLINE,///< Stopline in Block Memory. This line is included when running the Block Memory (default 0x0). The maximum value is 0x1FF.
  PCIE_BLOCKGEN_REPEAT,///< If set to 0, it means Infinite run. if set to 1, it means execute once. (default 0x1)
  PCIE_BLOCKGEN_ENABLED///< Enables/Disables Block Memory (default 0x1 for PCIE_HWCHANNEL_FUNCTION_A. For other hardware channels, the default is 0x0)
} EPCIEBlockGen;

/// \brief PCIE Exerciser properties for requester beahviour
///
typedef DEV_MIDL_ENUM enum EPCIEReqBehGen
{
  PCIE_REQBEHGEN_STARTLINE,///< Startline in Requester Behaviour Memory (default 0x0). See property \ref PCIE_BLOCKGEN_STARTLINE for possible use. The maximum value is the line specified in \ref PCIE_REQBEHGEN_STOPLINE.
  PCIE_REQBEHGEN_STOPLINE,///< Stopline in Requester Behaviour Memory. This line is included when running the Requester Behaviour Memory (default 0x0). The maximum value is 0x1FF.
  PCIE_REQBEHGEN_OBSOLETE_02
} EPCIEReqBehGen;

/// \brief PCIE Exerciser properties for completer beahviour
///
typedef DEV_MIDL_ENUM enum EPCIECompBehGen
{
  PCIE_COMPBEHGEN_STARTLINE,///< Startline in Completer Behaviour Memory (default 0x0). See property \ref PCIE_BLOCKGEN_STARTLINE for possible use. The maximum value is the line specified in \ref PCIE_COMPBEHGEN_STOPLINE.
  PCIE_COMPBEHGEN_STOPLINE,///< Stopline in Completer Behaviour Memory. This line is included when running the Completer Behaviour Memory (default 0x0). The maximum value is 0x1FF.
  PCIE_COMPBEHGEN_OBSOLETE_02
} EPCIECompBehGen;



/// \brief VC Settings
///
typedef DEV_MIDL_ENUM enum EPCIEVC
{
  PCIE_VC_TCVCMAP///< TC/VC Map of the Virtual Channel. Use the same format as "VC Resource Control Register" (refer Spec 1.0a, Section 7.11.7). This field indicates the TCs that are mapped to the VC resource. Bit locations within this field correspond to TC values. For example, when bit 7 is set in this field, TC7 is mapped to this VC resource. When more than one bit is set in this field, it indicates that multiple TCs are mapped to the VC resource.
} EPCIEVC;

typedef DEV_MIDL_ENUM enum EPCIEVH
{
  PCIE_VH1,
  PCIE_VH2,
  PCIE_VH3,
  PCIE_VH4,
} EPCIEVH;

/// \brief Config Space Settings
///
typedef DEV_MIDL_ENUM enum EPCIEConfigSpace
{
  PCIE_CONFIGSPACE_VCC_EXTVCCOUNT, ///< Virtual Channel Capability: Extended VC Count (Port VC Capability Reg 1) (maximum 0x7)
  PCIE_CONFIGSPACE_VCC_LPEXTVCCOUNT,///< Virtual Channel Capability: Low Priority Extended VC Count (Port VC Capability Reg 1 (max 0x7)
  PCIE_CONFIGSPACE_PCIECS_DEVPORTTYPE,///< PCI Express Capability Structure: Device Port Type (Device Capabilities Reg) (max 0xf)
  PCIE_CONFIGSPACE_HDR_MASTER_ENABLE,///< Configuration Space Header: Bus Master Enable
  PCIE_CONFIGSPACE_HDR_IO_ENABLE,///< Configuration Space Header: I/O Access Enable
  PCIE_CONFIGSPACE_HDR_MEM_ENABLE,///< Configuration Space Header: Memory Access Enable
  PCIE_CONFIGSPACE_HDR_EXPROM_ENABLE ///< Configuration Space Header: Address Decode Enable (Bit 0 in Expansion Rom Base Address)
} EPCIEConfigSpace;

/// \brief Exerciser Status

// enums used in callbacks

///
/// 
typedef DEV_MIDL_ENUM enum EPCIEExerciserStatusEvent
{
  PCIE_EXERCISERSTATUSEVENT_DATALINKSTATE,             ///< Data Link State changed. The value given is the new link state. (see \ref EPCIEDataLinkState) 
  PCIE_EXERCISERSTATUSEVENT_LINKWIDTH,                 ///< The negotiated link width. The value is Updated with each linkUp.
  PCIE_EXERCISERSTATUSEVENT_SI_SUCCESSFULLY_SENT,      ///< If the value is 1, it indicates that a send immediate packet was successfully sent. If the value is 0, it indicates that an error occured while trying to send a send immediate packet.
  PCIE_EXERCISERSTATUSEVENT_SI_COMPLETION_AVAILABLE,   ///< Indicates that there is a send immediate completion available.
  PCIE_EXERCISERSTATUSEVENT_UNEXPECTED_CRS_COMP,        // Unexpected CRS completion received
  PCIE_EXERCISERSTATUSEVENT_COMP_OTHER_THAN_SUCC_OR_CRS,// Completion status other than successful OR CRS
  PCIE_EXERCISERSTATUSEVENT_UNEXPECTED_COMP,           ///< Unexpected completion received. ATTENTION: This is obsolete for Gen3. Please use the new event class \ref ProtocolErrorStatusChanged.
  PCIE_EXERCISERSTATUSEVENT_COMP_TIMEOUT,              ///< Completion timeout. ATTENTION: This is obsolete for Gen3. Please use the new event class that \ref ProtocolErrorStatusChanged.
  PCIE_EXERCISERSTATUSEVENT_UNSUPPORTED_REQUEST,       ///< Unsupported request. ATTENTION: This is obsolete for Gen3. Please use the new event class \ref ProtocolErrorStatusChanged
  PCIE_EXERCISERSTATUSEVENT_DEC_MISS,                  ///< Decoder miss. ATTENTION: This is obsolete for Gen3. Please use the new event class \ref ProtocolErrorStatusChanged.
  PCIE_EXERCISERSTATUSEVENT_INVALID_REQUEST,           // invalid request
  PCIE_EXERCISERSTATUSEVENT_RX_POLARITY_REVERSAL,      ///< Reversed lanes after link negotiation, Bit0 = Physical Lane 0. Updated with each linkUp.
  PCIE_EXERCISERSTATUSEVENT_BLOCKMEM_RUN_STATE_CHANGED,///< Running state of the block memory changed.
  PCIE_EXERCISERSTATUSEVENT_INVALID_CLOCK,             // CAUTION: This event must be moved to the AgtControl interface (ModuleWrite() is in the AgtControl interface) !!!!! The IO Module has no valid clock. When using external clock make, sure it is available. Otherwise, use the internal clock source. 1=clock has changed to invalid, 0=clock has changed to valid
  PCIE_EXERCISERSTATUSEVENT_PROTERR,                   ///< Protocol error occurred. Masked protocol rules have no effect. If you receive this event, you should at least mask the passed rule (parameter status in IAgtPCIEExerciserEvents::ExerciserStatusChanged()) by calling ProtocolErrorMaskSet(). If you need an event with more detailed information about protocol error status changes, please refer to \ref ProtocolErrorStatusChanged.
  PCIE_EXERCISERSTATUSEVENT_TRIGGEROUT,                ///< Triggerout source mask or status has changed.
  PCIE_EXERCISERSTATUSEVENT_DATAGENCMP_ERROR_OCCURRED, ///< This indicates that a data generator compare error occurred.
  PCIE_EXERCISERSTATUSEVENT_DATAMEMCMP_ERROR_OCCURRED, ///< Memory mismatch for data compare occurred for one of the functions.
  PCIE_EXERCSIERSTATUSEVENT_POWERSTATE,                ///< Powerstate changed. Values are: 0=D0, 1=D2, 2=D2, and 3=D3.
  PCIE_EXERCISERSTATUSEVENT_LINKSPEED,                 ///< Link speed changed. Values are: 1=2.5 Gb/s, 2=5Gb/s and  4=8 Gb/s.
  PCIE_EXERCISERSTATUSEVENT_DATARATE_RECEIVED,         ///< This is the supported data rate information of the 'Data Rate Identifier' of incoming Training Sequences. It is updated each time the LTSSM enters the states 'Configuration.Complete' and 'Recovery.RcvrCfg'. The coding is as follows: Bit0 = Reserved, set to 0. Bit1 = generation 1 (2.5 Gb/s) data rate supported. Bit2 = generation 2 (5 Gb/s) data rate supported. Bit3 = generation 3 (8 Gb/s) data rate supported.
  PCIE_EXERCISERSTATUSEVENT_DATARATE_ADVERTISED,       ///< This is the supported data rate information of the 'Data Rate Identifier' of outgoing Training Sequences. It indicates the supported link speed. On Speed change from 5.0 Gb/s to 2.5 Gb/s, the advertised data rate is 2.5 Gb/s. It is restored back to Supported link speed when the link is retrained. The configuration is as follows: Bit0 = Reserved, set to 0. Bit1 = generation 1 (2.5 Gb/s) data rate supported. Bit2 = generation 2 (5 Gb/s) data rate supported. Bit3 = generation 3 (8 Gb/s) data rate supported
  PCIE_EXERCISERSTATUSEVENT_USBCOMMUNICATION_ERROR  ,  ///< There was an error while communicating with the USB Device.
  PCIE_EXERCISERSTATUSEVENT_FC_REQ_RECIEVED,
  PCIE_EXERCISERSTATUSEVENT_VC_RESOURCE_1_INITIALIZED, ///< VC Resource 1 initialized. 
  PCIE_EXERCISERSTATUSEVENT_MRIOV_VH1_INITIALIZED,     ///< VH 1 initialized. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*).
  PCIE_EXERCISERSTATUSEVENT_MRIOV_VH2_INITIALIZED,     ///< VH 2 initialized. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*).
  PCIE_EXERCISERSTATUSEVENT_MRIOV_VH3_INITIALIZED,     ///< VH 3 initialized. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*).
  PCIE_EXERCISERSTATUSEVENT_MRIOV_NEG ,                ///< MRIOV negotiated. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*).
  PCIE_EXERCISERSTATUSEVENT_MRIOV_VH4_INITIALIZED,     ///< VH 4 initialized. Applicable in the MRIOV mode only. (*New in Gen3 Exerciser*).
  PCIE_EXERCISERSTATUSEVENT_RX_PRESET_HINT,            ///< Rx Preset Hint value.
  PCIE_EXERCISERSTATUSEVENT_TX_PRESET_COEFF_CHANGED,   ///< Preset and Coefficient values has changed
  PCIE_EXERCISERSTATUSEVENT_INT_OPTION_CHANGED,        ///< The value of \ref PCIE_EXERCISER_STORAGE_INTERRUPT_OPTION changed. This is not supported as of now.
  PCIE_EXERCISERSTATUSEVENT_DUT_CFG_REG_CHANGED,       ///< The value of a DUT config register changed. This is not supported as of now.
  PCIE_EXERCISERSTATUSEVENT_NVME_ADMIN_QUEUE_INIT_DONE, ///< The value of Admin Queue Init done changed. '0' indicates init not done and '1' indicates init done.
  PCIE_EXERCISERSTATUSEVENT_NVME_CONT_REG_CHANGED,     ///< The value of an NVMe Controller Register changed.
  PCIE_EXERCISERSTATUSEVENT_NVME_SQ_COUNT_CHANGED,     ///< The number of Submission Queues changed.
  PCIE_EXERCISERSTATUSEVENT_NVME_CQ_COUNT_CHANGED,     ///< The number of Completion Queues changed.
  PCIE_EXERCISERSTATUSEVENT_NVME_SQ_DOORBELL_CHANGED,  ///< The Doorbell of a Submission Queue changed.
  PCIE_EXERCISERSTATUSEVENT_NVME_CQ_DOORBELL_CHANGED,  ///< The Doorbell of a Completion Queue changed.
  PCIE_EXERCISERSTATUSEVENT_NVME_CPL_RECEIVED          ///< NVMe Completion Received. This is not supported as of now.

} EPCIEExerciserStatusEvent;

//typedef DEV_MIDL_ENUM enum EPCIEDecoderChange
//{
//  PCIE_DECODERCHANGE_
//} EPCIEDecoderChange;

/// \brief Link State Settings
typedef DEV_MIDL_ENUM enum EPCIELinkStateDirect
{
  PCIE_LINKSTATEDIRECT_RECOVERY,      ///< Link State is forced in the Recovery state.
  PCIE_LINKSTATEDIRECT_LOOPBACK_FROM_CONFIGURATION, ///< As soon as the LTSSM reaches the state 'Configuration', it is directed into the state 'Loopback'. 
  PCIE_LINKSTATEDIRECT_LOOPBACK_FROM_L0,            ///< As soon as the LTSSM reaches the state 'L0', it is directed into the state 'Loopback' via 'Recovery'. 
  PCIE_LINKSTATEDIRECT_DISABLED_FROM_CONFIGURATION, ///< As soon as the LTSSM reaches the state 'Configuration', it is directed into the state 'Disabled'. 
  PCIE_LINKSTATEDIRECT_DISABLED_FROM_L0,            ///< As soon as the LTSSM reaches the state 'L0', it is directed into the state 'Disabled' via 'Recovery'. 
  PCIE_LINKSTATEDIRECT_HOT_RESET,                   ///< As soon as the LTSSM reaches the state 'L0', it is directed into the state 'Hot Reset' via 'Recovery'. 
  PCIE_LINKSTATEDIRECT_CONFIGURATION,               ///< As soon as the LTSSM reaches the state 'L0', it is directed into the state 'Configuration' via 'Recovery'. 
  PCIE_LINKSTATEDIRECT_IDLE,                        ///< The LTSSM is directed to an idle state and remains there until \ref PCIE_LINKSTATEDIRECT_START_LINKTRAINING is asserted. 
  PCIE_LINKSTATEDIRECT_START_LINKTRAINING,          ///< The link training is started. To return to an idle state, use \ref PCIE_LINKSTATEDIRECT_IDLE. 
  PCIE_LINKSTATEDIRECT_SPEED_CHANGE,                 
  ///< From the state 'L0', this will initiate a speed change from the current speed to the highest advertized data rate by both Exerciser and DUT.
  ///Speed change to a lower data rate would be done by advertizing lower data rate as Data Rate Capability. 
  ///Please refer to ExerciserSet, property PCIE_EXERCISER_DATA_RATE to modify the Link Speed capability. 
  ///Please be also aware that the definition of 'success' for this direct command is that the LTSSM has seen 
  ///Recovery.Speed' state. The success of this command does not necessarily mean that the link speed really changed. 
  ///Use ExerciseStatusRead to retrieve the current link speed. For example, if the link partner does not support 
  ///5.0 GT/s or 8 GT/s, the status for this command will show 'success' but the link is still at 2.5 GT/s speed. 
  PCIE_LINKSTATEDIRECT_CONFIGURATION_UPCONFIGURE,   ///< As soon as the LTSSM reaches the state 'L0', it is directed into the state 'Configuration' via 'Recovery'. In this state, the Exerciser will UpConfigure to its originial linkwidth. 
  PCIE_LINKSTATEDIRECT_CONFIGURATION_DOWNCONFIGURE  ///< As soon as the LTSSM reaches the state 'L0', it is directed into the state 'Configuration' via 'Recovery'. In this state, the Exerciser will DownConfigure to x1. 
} EPCIELinkStateDirect;


/// \brief Rx Error Settings
///
/// The link partner will retry sending the packet with the sequence number 'PCIE_RXERROR_NAK_SEQUENCE_NUM' until it receives an ACK (which is after 'PCIE_RXERROR_NAK_COUNT'+1 times). 
/// As soon as the 'PCIE_RXERRORSTATUS_NAK_COUNT' counter in 'EPCIERxErrorStatus' reaches zero, the behavior must be restarted again with 'RxErrorStart'.
/// Please be aware that due to the usage of the 'NAK scheduled flag' (see PCI Express Specification), it is possible that not all NAK packets are actually sent.
typedef DEV_MIDL_ENUM enum EPCIERxError
{
  PCIE_RXERROR_NAK_SEQUENCE_NUM,///< For packets with this sequence number, the hw responds with a NAK instead of an ACK. (maximum value is 0xfff)
  PCIE_RXERROR_NAK_COUNT///< Number of times the hw responds with a NAK instead of an ACK for the sequence number 'PCIE_RXERROR_NAK_SEQUENCE_NUM'. (maximum value is 0xf)
  //PCIE_RXERROR_NAK_SEQUENCE_MASK 
} EPCIERxError;

/// \brief Rx Error Status information
typedef DEV_MIDL_ENUM enum EPCIERxErrorStatus
{
  PCIE_RXERRORSTATUS_NAK_COUNT///< Remaining number of packets the hw will respond to with NAK. The maximum value is PCIE_RXERROR_NAK_COUNT in \ref EPCIERxError, the minimum value is 0.
} EPCIERxErrorStatus;

/// \brief Returns exerciser status information
typedef DEV_MIDL_ENUM enum EPCIEExerciserStatus 
{
  PCIE_EXERCISERSTATUS_OBSOLETE_00,
  PCIE_EXERCISERSTATUS_OBSOLETE_01,
  PCIE_EXERCISERSTATUS_OBSOLETE_02,
  PCIE_EXERCISERSTATUS_OBSOLETE_03,
  PCIE_EXERCISERSTATUS_OBSOLETE_04,
  PCIE_EXERCISERSTATUS_OBSOLETE_05,
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_INTADDR,          ///< This is the address in the data memory for Function A where the error occurred. PCIE_EXERCISERSTATUS_DATAMEMCMP_INTADDR is deprecated. Data compare can now be used on a per function basis. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_DATA_EXPECTED,    ///< This is the expected byte of the data compare for Function A. PCIE_EXERCISERSTATUS_DATAMEMCMP_DATA_EXPECTED is deprecated. Data compare can now be used on a per function basis. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_DATA_REAL,        ///< This is the actual byte of the data compare for Function A. PCIE_EXERCISERSTATUS_DATAMEMCMP_DATA_REAL is deprecated. Data compare can now be used on a per function basis. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_OBSOLETE_09,
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_ERROR_OCCURRED,   ///< This indicates that a data compare error occurred for Function A. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. PCIE_EXERCISERSTATUS_DATAMEMCMP_ERROR_OCCURRED is deprecated. Data compare can now be used on a per function basis. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_ERROR_COUNT,      ///< Number of incorrect bytes found in data comparison for Function A. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. PCIE_EXERCISERSTATUS_DATAMEMCMP_ERROR_COUNT is deprecated. Data compare can now be used on a per function basis. (*New in Gen3 Exerciser*)

  //PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER0,                      ///< First DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER1,                      ///< Second DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER2,                      ///< Third DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER3,                      ///< Fourth DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_DWOFFSET,                     ///< Zero-based DW offset into packet where error occurred. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_DATA_EXPECTED,                ///< This is the expected DW of the data generator compare. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_DATA_REAL,                    ///< This is the actual DW of the data generator compare. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_BYTEN,                        ///< These are the actual byte enbles of the data generator compare. Please use \ref ExerciserStatusReset to reset this information.
  //PCIE_EXERCISERSTATUS_DATAGENCMP_ERROR_OCCURRED,               ///< This indicates that a data generator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.

  PCIE_EXERCISERSTATUS_OUTSTANDING_REQUESTS,                    ///< This is the number of outstanding requests of the Exerciser.
  PCIE_EXERCISERSTATUS_RX_POLARITY_INVERSION,                   ///< Lanes that negotiated a polarity inversion. Bit0 is Physical Lane 0, Bit1 is Physical Lane 1 etc. Updated with each linkUp. 
  PCIE_EXERCISERSTATUS_REVERSED_LANES,                          ///< Indicates whether the neogtiated link has reversed lanes. For example, for a negotiated link width of 4, this will result in a lane order of 3,2,1,0 instead of 0,1,2,3.
  PCIE_EXERCISERSTATUS_HWCHANNEL_A_STATE,                       ///< Current state of the Hardware Channel A State Machine. For return values, see \ref EPCIEHWChannelStateVal.
  PCIE_EXERCISERSTATUS_HWCHANNEL_B_STATE,                       ///< Current state of the Hardware Channel B State Machine. For return values, see \ref EPCIEHWChannelStateVal.
  PCIE_EXERCISERSTATUS_HWCHANNEL_C_STATE,                       ///< Current state of the Hardware Channel C State Machine. For return values, see \ref EPCIEHWChannelStateVal.
  PCIE_EXERCISERSTATUS_HWCHANNEL_D_STATE,                       ///< Current state of the Hardware Channel D State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_E_STATE,                       ///< Current state of the Hardware Channel E State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_BVF1_STATE,                    ///< Current state of the Hardware Channel BVF1 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_BVF2_STATE,                    ///< Current state of the Hardware Channel BVF2 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_CVF1_STATE,                    ///< Current state of the Hardware Channel CVF1 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_CVF2_STATE,                    ///< Current state of the Hardware Channel CVF2 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_DVF1_STATE,                    ///< Current state of the Hardware Channel DVF1 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_DVF2_STATE,                    ///< Current state of the Hardware Channel DVF2 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_EVF1_STATE,                    ///< Current state of the Hardware Channel EVF1 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_EVF2_STATE,                    ///< Current state of the Hardware Channel EVF2 State Machine. For return values, see \ref EPCIEHWChannelStateVal.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_INVALID_CLOCK,                           ///< The IO Module has no valid clock. When using external clock, make sure it is available. Otherwise, use the internal clock source.
  PCIE_EXERCISERSTATUS_PROTERR,                                 ///< Protocol error occurred. Masked protocol rules have no effect on this bit. Cleared by ProtocolErrorStatusClear().
  PCIE_EXERCISERSTATUS_LTSSM_STATE,                             ///< Current state of the transmitter part of the Link Training Status State Machine (LTSSM). The value is updated when the state is entered. For interpretation of the value, please refer to \ref EPCIELTSSMVal.
  PCIE_EXERCISERSTATUS_LTSSM_VERSION,                           // Version of the LTSSM currently loaded. The value returned is 32 bit which means each information is 1 byte, from left to right(MSB to LSB): <OperationMode>.<LTSSMNumber>.<MajorVersion>.<MinorVersion>. OperationMode is Debug (0), Normal (1) TestCase (2). Returning zero for all information means that no LTSSM is loaded. 
  PCIE_EXERCISERSTATUS_LINKSTATE_DIRECT_PENDING,                ///< This returns 1 if a call to \ref LinkStateDirect has not yet been accomplished successfully. 
  PCIE_EXERCISERSTATUS_POWERSTATE,                              ///< Current powerstate. Values are: 0=D0, 1=D2, 2=D2, and 3=D3.
  PCIE_EXERCISERSTATUS_LINKSPEED,                               ///< Current link speed. Values are: 1=2.5 Gb/s , 2=5Gb/s and 4=8Gb/s.
  PCIE_EXERCISERSTATUS_HARDWARE,                                ///< Reports the connected hardware. '1': Gen1 device (E2960A), '2': Gen2 device (E2960B) and '3': Gen3 device (U4305A).
  PCIE_EXERCISERSTATUS_LAST_TRAINING_CONTROL,                   ///< This is the Training Control (Symbol Number 5) field of the last TS seen on lane 0. 
  PCIE_EXERCISERSTATUS_DATARATE_RECEIVED,                       ///< This is the supported data rate information of the 'Data Rate Identifier' of incoming Training Sequences. It is updated each time the LTSSM enters the states 'Configuration.Complete' and 'Recovery.RcvrCfg'. The coding is as follows: Bit0 = Reserved, set to 0. Bit1 = generation 1 (2.5 Gb/s) data rate supported. Bit2 = generation 2 (5 Gb/s) data rate supported. Bit3 = generation 3 (8 Gb/s) data rate supported.
  PCIE_EXERCISERSTATUS_DATARATE_ADVERTISED,                     ///< This is the supported data rate information of the 'Data Rate Identifier' of outgoing Training Sequences. It indicates the supported link speed. On Speed change from 5.0 Gb/s to 2.5 Gb/s, the advertised data rate is 2.5 Gb/s. It is restored back to the supported link speed when the link is retrained. The coding is as follows: Bit0 = Reserved, set to 0. Bit1 = generation 1 (2.5 Gb/s) data rate supported. Bit2 = generation 2 (5 Gb/s) data rate supported. Bit3 = generation 3 (8 Gb/s) data rate supported
  PCIE_EXERCISERSTATUS_UPCONFIGURE_CAPABILITY_RECEIVED,         ///< This is the upconfigure capability information shown by the incoming Training Sequences. 
  PCIE_EXERCISERSTATUS_FC_REQ_RECIEVED,                         ///< Per VH Flow control requested by peer device. (*New in Gen3 Exerciser*)
                                                                /// \li '0' -> Peer device request VL based flow control
                                                                /// \li '1' -> Peer device request per VH flow control
  PCIE_EXERCISERSTATUS_VC_RESOURCE_1_INITIALIZED,               ///< Indicates whether or not the Virtual Channel Resource 1 has initialized. Applicable in the PCIe/SRIOV mode only (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_MRIOV_VH1_INITIALIZED,                   ///< Indicates whether or not the Virtual Hierarchy 1 initialized. Applicable only in the MRIOV mode (*New in Gen3 Exerciser*)
                                                                /// \li '0' -> Not Initialized/Initialization Pending 
                                                                /// \li '1' -> Initialized
  PCIE_EXERCISERSTATUS_MRIOV_VH2_INITIALIZED,                   ///< Indicates whether or not the Virtual Hierarchy 2 initialized. Applicable only in the MRIOV mode (*New in Gen3 Exerciser*)
                                                                /// \li '0' -> Not Initialized/Initialization Pending 
                                                                /// \li '1' -> Initialized
  PCIE_EXERCISERSTATUS_MRIOV_VH3_INITIALIZED,                   ///< Indicates whether or not the Virtual Hierarchy 3 initialized. Applicable only in the MRIOV mode (*New in Gen3 Exerciser*)
                                                                /// \li '0' -> Not Initialized/Initialization Pending 
                                                                /// \li '1' -> Initialized
  PCIE_EXERCISERSTATUS_MRIOV_NEG,                               ///< Represents which protocol mode is negotiated. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_A_OUTSTANDING_REQUESTS,        ///< This is the number of outstanding requests of Hardware Channel for Function A.
  PCIE_EXERCISERSTATUS_HWCHANNEL_B_OUTSTANDING_REQUESTS,        ///< This is the number of outstanding requests of Hardware Channel for Function B.
  PCIE_EXERCISERSTATUS_HWCHANNEL_C_OUTSTANDING_REQUESTS,        ///< This is the number of outstanding requests of Hardware Channel for Function C.
  PCIE_EXERCISERSTATUS_HWCHANNEL_D_OUTSTANDING_REQUESTS,        ///< This is the number of outstanding requests of Hardware Channel for Function D.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_E_OUTSTANDING_REQUESTS,        ///< This is the number of outstanding requests of Hardware Channel for Function E.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_BVF1_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function B VF1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_BVF2_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function B VF2.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_CVF1_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function C VF1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_CVF2_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function C VF2.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_DVF1_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function D VF1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_DVF2_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function D VF2.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_EVF1_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function E VF1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_HWCHANNEL_EVF2_OUTSTANDING_REQUESTS,     ///< This is the number of outstanding requests of Hardware Channel for Function E VF2.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_MRIOV_VH4_INITIALIZED,                   ///< Indicates whether or not the Virtual Hierarchy 4 initialized. Applicable only in the MRIOV mode (*New in Gen3 Exerciser*)
                                                                /// \li '0' -> Not Initialized/Initialization Pending 
                                                                /// \li '1' -> Initialized
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_INTADDR,          ///< This is the address in the data memory for Function B where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_INTADDR,          ///< This is the address in the data memory for Function C where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_INTADDR,          ///< This is the address in the data memory for Function D where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_INTADDR,          ///< This is the address in the data memory for Function E where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_INTADDR,       ///< This is the address in the data memory for Function B VF1 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_INTADDR,       ///< This is the address in the data memory for Function B VF2 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_INTADDR,       ///< This is the address in the data memory for Function C VF1 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_INTADDR,       ///< This is the address in the data memory for Function C VF2 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_INTADDR,       ///< This is the address in the data memory for Function D VF1 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_INTADDR,       ///< This is the address in the data memory for Function D VF2 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_INTADDR,       ///< This is the address in the data memory for Function E VF1 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_INTADDR,       ///< This is the address in the data memory for Function E VF2 where the error occurred. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_DATA_EXPECTED,    ///< This is the expected byte of the data compare for Function B. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_DATA_EXPECTED,    ///< This is the expected byte of the data compare for Function C. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_DATA_EXPECTED,    ///< This is the expected byte of the data compare for Function D. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_DATA_EXPECTED,    ///< This is the expected byte of the data compare for Function E. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function B VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function B VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function C VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function C VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function D VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function D VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function E VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_DATA_EXPECTED, ///< This is the expected byte of the data compare for Function E VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_DATA_REAL,        ///< This is the actual byte of the data compare for Function B. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_DATA_REAL,        ///< This is the actual byte of the data compare for Function C. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_DATA_REAL,        ///< This is the actual byte of the data compare for Function D. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_DATA_REAL,        ///< This is the actual byte of the data compare for Function E. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_DATA_REAL,     ///< This is the actual byte of the data compare for Function B VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_DATA_REAL,     ///< This is the actual byte of the data compare for Function B VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_DATA_REAL,     ///< This is the actual byte of the data compare for Function C VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_DATA_REAL,     ///< This is the actual byte of the data compare for Function C VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_DATA_REAL,     ///< This is the actual byte of the data compare for Function D VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_DATA_REAL,     ///< This is the actual byte of the data compare for Function D VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_DATA_REAL,     ///< This is the actual byte of the data compare for Function E VF1. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_DATA_REAL,     ///< This is the actual byte of the data compare for Function E VF2. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_ERROR_OCCURRED,   ///< This indicates that a data compare error occurred for Function B. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_ERROR_OCCURRED,   ///< This indicates that a data compare error occurred for Function C. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_ERROR_OCCURRED,   ///< This indicates that a data compare error occurred for Function D. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_ERROR_OCCURRED,   ///< This indicates that a data compare error occurred for Function E. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function B VF1. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function B VF2. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function C VF1. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function C VF2. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function D VF1. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function D VF2. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function E VF1. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_ERROR_OCCURRED,///< This indicates that a data compare error occurred for Function E VF2. Please use \ref ExerciserStatusReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_ERROR_COUNT,      ///< Number of incorrect bytes found in data comparison for Function B. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_ERROR_COUNT,      ///< Number of incorrect bytes found in data comparison for Function C. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_ERROR_COUNT,      ///< Number of incorrect bytes found in data comparison for Function D. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_ERROR_COUNT,      ///< Number of incorrect bytes found in data comparison for Function E. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function B VF1. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function B VF2. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function C VF1. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function C VF2. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function D VF1. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function D VF2. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function E VF1. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_ERROR_COUNT,   ///< Number of incorrect bytes found in data comparison for Function E VF2. \ref ExerciserStatusSnapshot or \ref ExerciserStatusFunctionSnapshot needs to be called before reading this value. Please use \ref ExerciserStatusReset or \ref ExerciserStatusFunctionReset to reset this information. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_RX_PRESET_HINT,                          ///< Rx Preset Hint
  PCIE_EXERCISERSTATUS_DUT_FS,									///< DUTs FS captured by Exerciser 
  PCIE_EXERCISERSTATUS_DUT_LF,									///< DUTs LF captured by Exerciser 
} EPCIEExerciserStatus;

/// \brief Returns exerciser phy status information
/// \use IAgtPCIEExerciser::ExerciserPhyStatusRead
typedef DEV_MIDL_ENUM enum EPCIEExerciserPhyStatus 
{
  PCIE_EXERCISERPHYSTATUS_TRANSMITTER_SETTING_RECEIVED_PRESET,
  ///< This is the number of outstanding requests of Hardware Channel for Function C.
  PCIE_EXERCISERPHYSTATUS_TRANSMITTER_SETTING_RECEIVED_C_MINUS_1,
  ///< This is the Transmitter settings received C-1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_TRANSMITTER_SETTING_RECEIVED_C_ZERO,
  ///< This is the Transmitter settings received C0. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_TRANSMITTER_SETTING_RECEIVED_C_PLUS_1,
  ///< This is the Transmitter settings received C+1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_TRANSMITTER_SETTING_RECEIVED_REJECT,
  ///< This is the Transmitter settings received Preset/Coefficient Rejected.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_COEFFICIENT_REQUEST_FROM_EXERCISER_PRESET,
  ///< This is the nCoefficient Request From Exerciser Preset(*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_COEFFICIENT_REQUEST_FROM_EXERCISER_C_MINUS_1,
  ///< This is the Coefficient Request From Exerciser C-1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_COEFFICIENT_REQUEST_FROM_EXERCISER_C_ZERO,
  ///< This is the Coefficient Request From Exerciser C0. (*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_COEFFICIENT_REQUEST_FROM_EXERCISER_C_PLUS_1,
  ///< This is the Coefficient Request From Exerciser C+1.(*New in Gen3 Exerciser*)
  PCIE_EXERCISERPHYSTATUS_COEFFICIENT_REQUEST_FROM_EXERCISER_REJECT,
  ///< This is the Coefficient Request From Exerciser Preset/Coefficient Rejected.(*New in Gen3 Exerciser*)

} EPCIEExerciserPhyStatus;

/// \brief Returns exerciser status information for specified function
/// \use IAgtPCIEExerciser::ExerciserChannelFunctionGet or
/// \use IAgtPCIEExerciser::ExerciserChannelFunctionSet
typedef DEV_MIDL_ENUM enum EPCIEExerciserHwChannelStatus
{
  PCIE_EXERCISER_GENERATOR_PREFIX,
  ///< Bits[31:21] of this value are used for the datagenerator.
  /// Bits[20:0] of this value are ignored (are taken from the bus address in the datagenerator). Read & Write (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER_PREFIX,
  ///< Prefix of the packet header when datagenerator compare error occurred, Valid for MRIOV mode only. Please use \ref ExerciserStatusReset to reset this information. Read only (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER0,
  ///< First DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information. Read only  (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER1, 
  ///< Second DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.  Read only (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER2,
  ///< Third DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information.  Read only (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_HEADER3,
  ///< Fourth DW of packet header when datagenerator compare error occurred. Please use \ref ExerciserStatusReset to reset this information. Read only  (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_DWOFFSET,
  ///< Zero-based DW offset into packet where error occurred. Please use \ref ExerciserStatusReset to reset this information.  Read only (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_DATA_EXPECTED,
  ///< This is the expected DW of the data generator compare. Please use \ref ExerciserStatusReset to reset this information. Read only  (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_DATA_REAL,
  ///< This is the actual DW of the data generator compare. Please use \ref ExerciserStatusReset to reset this information. Read only  (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_BYTEN, 
  ///< These are the actual byte enbles of the data generator compare. Please use \ref ExerciserStatusReset to reset this information. Read only (*New in Gen3 Exerciser*)
  PCIE_EXERCISERSTATUS_DATAGENCMP_ERROR_OCCURRED
  ///< This indicates that a data generator compare error occurred. Please use \ref ExerciserStatusReset to reset this information. Read only (*New in Gen3 Exerciser*)

} EPCIEExerciserHwChannelStatus;

/// \brief Values of the Hardware Channel State Machine
///
/// The numbers indicated in parentheses are the values returned when reading the Hardware Channel states via \ref ExerciserStatusRead
typedef DEV_MIDL_ENUM enum EPCIEHWChannelStateVal
{
  PCIE_HWCHANNELSTATEV_STOPPED  = 0, ///< (0) The Hardware Channel stopped.
  PCIE_HWCHANNELSTATEV_RUNNING  = 1, ///< (1) The Hardware Channel is running.
  PCIE_HWCHANNELSTATEV_INVALID  = 2, ///< (2) The Hardware Channel is in an invalid state.
  PCIE_HWCHANNELSTATEV_STOPPING = 3, ///< (3) The Hardware Channel is stopping.
} EPCIEHWChannelStateVal;

/// \brief Pattern
typedef DEV_MIDL_ENUM enum EPCIEPattern
{
  PCIE_PATTERN_LEN = 0,  ///< Length of data payload in DWORD (bits [1:0] of byte 2, bits [7:0] of byte 3) (default 0x0) 
  PCIE_PATTERN_RSVD023,  ///< Reserved (bits [3:2] of byte 2) (default 0x0)
  PCIE_PATTERN_ATTR,     ///< Attributes (bits [5:4] of byte 2) (default 0x0)
  PCIE_PATTERN_EP,       ///< TLP is poisoned (bit 6 of byte 2) (default 0x0)
  PCIE_PATTERN_TD,       ///< TLP digest present (bit 7 of byte 2) (default 0x0)
  PCIE_PATTERN_RSVD013,  ///< Reserved (bits [3:0] of byte 1) (default 0x0)
  PCIE_PATTERN_TC,       ///< Traffic Class (bits [6:4] of byte 1) (default 0x0)
  PCIE_PATTERN_RSVD017,  ///< Reserved (bit 7 of byte 1) (default 0x0)
  PCIE_PATTERN_TYPE,     ///< Type of TLP (bits 4:0 of byte 0) (default 0x0)
  PCIE_PATTERN_FMT,      ///< Format of TLP (bits 6:5 of byte 0) (default 0x0)
  PCIE_PATTERN_RSVD007,  ///< Reserved (bit 7 of byte 0) (default 0x0)
  PCIE_PATTERN_1STDWBE,  ///< byte enables for the first (or only) DWORD (bits [3:0] of byte 7) (default 0x0)
  PCIE_PATTERN_LASTDWBE, ///< byte enables for the last DWORD (bits [7:4] of byte 7) (default 0x0)
  PCIE_PATTERN_TAG,      ///< Tag of the transaction (bits [7:0] of byte 6) (default 0x0)
  PCIE_PATTERN_REQID,    ///< Requester ID (bits [7:0] of byte 5, bits [7:0] of byte 4) (default 0x0)
  PCIE_PATTERN_CFG_RSVD111,  // Reserved (bits [1:0] of byte 11) (default 0x0)
  PCIE_PATTERN_CFG_REGNUM,   ///< Register Number (bits [7:2] of byte 11) (default 0x0)
  PCIE_PATTERN_CFG_EXTREGNUM,///< Extended Register Number (bits [3:0] of bytes 10) (default 0x0)
  PCIE_PATTERN_CFG_RSVD107,  ///< Reserved (bits [7:4] of byte 10) (default 0x0)
  PCIE_PATTERN_CFG_FUNCNUM,  ///< Function Number (bits [2:0] of byte 9) (default 0x0)
  PCIE_PATTERN_CFG_DEVNUM,   ///< Device Number (bits [7:3] of byte 9) (default 0x0)
  PCIE_PATTERN_CFG_BUSNUM,   ///< Bus Number (bits [7:0] of byte 8) (default 0x0)
  PCIE_PATTERN_MEM64_ADDRHI, ///< Address bits [63:32] in a 64-bit memory access (bytes 8,9,10,11) (default 0x0)
  PCIE_PATTERN_MEM64_ADDRLO, ///< Address bits [31:0] in a 64-bit memory access (bytes 12,13,14,15) (default 0x0)
  PCIE_PATTERN_MEM64_RSVD151,///< Reserved (bits [1:0] of byte 15) (default 0x0)
  PCIE_PATTERN_MEM32_ADDR,   ///< Address bits [31:0] in a 32-bit memory access (bytes 8,9,10,11) (default 0x0)
  PCIE_PATTERN_MEM32_RSVD111,// Reserved (bits [1:0] of byte 11) (default 0x0)
  PCIE_PATTERN_MCODE,        ///< Message code (bits [7:0] of byte 7) (default 0x0)
  PCIE_PATTERN_IO_ADDR,          ///< Address bits [31:0] in a 32-bit IO access (default 0x0)
  PCIE_PATTERN_IO_RSVD111,       ///< Reserved (bits [1:0] of byte 11) (default 0x0)    
  PCIE_PATTERN_MSGAS_RSVD047,    // Reserved for MsgAS (bytes 4,5,6,7) (default 0x0)
  PCIE_PATTERN_MSGAS_RSVD087,    // Reserved for MsgAS (bytes 8,9,10,11) (default 0x0)
  PCIE_PATTERN_MSGAS_RSVD127,    // Reserved for MsgAS (bytes 12,13,14,15) (default 0x0)
  PCIE_PATTERN_MSG_BYTES08_11,    // Fields dependent on type of message (bytes 8,9,10,11) (default 0x0)
  PCIE_PATTERN_MSG_BYTES12_15,    // Fields dependent on type of message (bytes 12,13,14,15) (default 0x0)
  PCIE_PATTERN_MSGASD_RSVD047,    // Reserved for MsgASD (bytes 4,5,6,7) (default 0x0)
  PCIE_PATTERN_MSGASD_RSVD087,    // Reserved for MsgASD (bytes 8,9,10,11) (default 0x0)
  PCIE_PATTERN_MSGASD_RSVD127,    // Reserved for MsgASD (bytes 12,13,14,15) (default 0x0)
  PCIE_PATTERN_COMP_REQID,///< Requester ID of the completion (bytes 8,9). (default 0x0)
  PCIE_PATTERN_COMP_COMPLID,///< Completion ID of the completion (bytes 4,5). (default 0x0)
  PCIE_PATTERN_COMP_STATUS,///< Completion status of the completion (bits[7:5] of byte 6) (default 0x0)
  PCIE_PATTERN_COMP_BYTECOUNT,///< Bytecount of the completion (bits[3:0] of byte 6 and byte 7) (default 0x0)
  PCIE_PATTERN_COMP_BCM,///< BCM of the completion (bit 4 of byte 6) (default 0x0)
  PCIE_PATTERN_COMP_TAG,///< Tag of the completion (byte 10) (default 0x0)
  PCIE_PATTERN_COMP_RSVD117,///< Reserved bits of the completion (bit 7 of byte 11) (default 0x0)
  PCIE_PATTERN_COMP_LOWERADDR,///< Lower address field of the completion (bits[6:0] of byte 11) (default 0x0)
  PCIE_PATTERN_HEADER_DW0,     ///< PCIe Header DWord 0
  PCIE_PATTERN_HEADER_DW1,     ///< PCIe Header DWord 1
  PCIE_PATTERN_HEADER_DW2,     ///< PCIe Header DWord 2
  PCIE_PATTERN_HEADER_DW3,     ///< PCIe Header DWord 3
  PCIE_PATTERN_PAYLOAD_DW0,    ///< Payload DWord 0
  PCIE_PATTERN_PAYLOAD_DW1,    ///< Payload DWord 1
  PCIE_PATTERN_PAYLOAD_DW2,    ///< Payload DWord 2
  PCIE_PATTERN_PAYLOAD_DW3,     ///< Payload DWord 3
  PCIE_PATTERN_MRIOV_PREFIX_HEADER,    ///< MRIOV Prefix Header DWord
  PCIE_PATTERN_MRIOV_PREFIX_PREFIXTAG, ///< MRIOV Prefix - Prefix Identifying Tag
  PCIE_PATTERN_MRIOV_PREFIX_GLOBALKEY, ///< MRIOV Prefix - Global Key
  PCIE_PATTERN_MRIOV_PREFIX_RSVD023,   ///< MRIOV Prefix - Reserved (bit 3 of byte 2) (default 0x0)
  PCIE_PATTERN_MRIOV_PREFIX_VLNUM,     ///< MRIOV Prefix - VL Number
  PCIE_PATTERN_MRIOV_PREFIX_VHNUM,     ///< MRIOV Prefix - VH Number
} EPCIEPattern;

/// \brief Returns performance counter status information (*Changed in Gen3 Exerciser*)
typedef DEV_MIDL_ENUM enum EPCIEPerformanceCounterStatus
{
  PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_CODING_ERR,
  ///< Number of coding errors on lane x within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN. 
  ///It is neccessary to set the lane number used in the measurement first before starting the measurement.
  ///This needs to be done with \ref ExerciserSet and enum PCIE_EXERCISER_PERFORMANCECOUNTER_DLLPHY_ERR_LANE.
  PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_DISPARITY_ERR,
  ///< Number of disparity errors on lane x within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN.
  ///It is neccessary to set the lane number used in the measurement first before starting the measurement.
  ///This needs to be done with \ref ExerciserSet and enum PCIE_EXERCISER_PERFORMANCECOUNTER_DLLPHY_ERR_LANE.
  PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_128B_130B_ERR,
  ///< Number of 128B/130B errors on lane x within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN.
  ///It is neccessary to set the lane number used in the measurement first before starting the measurement.
  ///This needs to be done with \ref ExerciserSet and enum PCIE_EXERCISER_PERFORMANCECOUNTER_DLLPHY_ERR_LANE.
  PCIE_PERFORMANCECOUNTERSTATUS_SUCCESSFUL_LINK_TRAININGS, 
  ///< Number of successful link trainings.
  PCIE_PERFORMANCECOUNTERSTATUS_DLLP_ACK, ///< Total number of 'ACK' Data Link Layer packets received. 
  PCIE_PERFORMANCECOUNTERSTATUS_DLLP_NAK, ///< Total number of 'NAK' Data Link Layer packets received. 
  PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN, ///< Time in ns
  PCIE_PERFORMANCECOUNTERSTATUS_DLLP_TX_NAK, ///< Total number of 'NAK' Data Link Layer packets transmitted.  
  ///< Performance counter recording interval in ns. 
  ///< Use PerformanceCounterStatusReset to start recording and PerformanceCounterStatusSnapshot to stop recording.
  ///< This information is independent of the TC/Lane number. 
} EPCIEPerformanceCounterStatus;

/// \brief Returns performance counter VC status information (*New in Gen3 Exerciser*)
typedef DEV_MIDL_ENUM enum EPCIEPerformanceCounterVCStatus
{
  PCIE_PERFORMANCECOUNTERSTATUS_DLLP_FC, ///< Total number of InitFC and UpdateFC packets received.
} EPCIEPerformanceCounterVCStatus;

/// \brief Returns performance counter Function status information (*New in Gen3 Exerciser*)
typedef DEV_MIDL_ENUM enum EPCIEPerformanceCounterFunctionStatus
{
  PCIE_PERFORMANCECOUNTERSTATUS_TLP_NUM,
  ///< Number of TLPs within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN.
  PCIE_PERFORMANCECOUNTERSTATUS_DW_NUM,
  ///< Number of DWords within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN.
  PCIE_PERFORMANCECOUNTERSTATUS_TX_TLP_NUM, 
  ///< Number of TLPs transmitted within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN.
  PCIE_PERFORMANCECOUNTERSTATUS_TX_DW_NUM, 
  ///< Number of DWords within the time interval PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN.
} EPCIEPerformanceCounterFunctionStatus;


/// \brief TriggerOut sources/masks
///
/// Describes the various Trigger Out sources/masks.
/// Use these masks to get the bits for the various sources in the accumulated and first trigger status registers.
typedef DEV_MIDL_ENUM enum EPCIETriggerOut
{
  // Please do not change the enum values without checking the HW registers !
  // The API calls TriggerOutSet/Get only work under the assumption that only one bit at a time is set
  // for each enum value!
  PCIE_TRIGGEROUT_PROTERR = 0x1,           ///< Protocol error
  PCIE_TRIGGEROUT_PATTERN = 0x4,           ///< Pattern terms. A trigger out event occurs if the result of any of the pattern matchers is true (PM1 || PM2 || PM3 || PM4). ATTTENTION: Please be aware that a pattern matcher with a mask set to 'don't care' for all bits is true for any packet !
  PCIE_TRIGGEROUT_CMPERR  = 0x8,           ///< Data compare error (datamem/datagen)
  PCIE_TRIGGEROUT_GOING_OUT_OF_L0 = 0x1000 ///< LTSSM was going out of L0.
} EPCIETriggerOut;


/// \brief Dllp control status 
///
/// \use IAgtPCIEExerciser::DllpStatusGet, IAgtPCIEExerciser::DllpStatusClear
typedef DEV_MIDL_ENUM enum EPCIEDllpStatus
{
  PCIE_DLLPSTATUS_DATASENT,
  ///< Returns 1 if DLLP has finally been sent after a call to IAgtPCIEExerciser::DllpDataSend.
  ///< Cannot be cleared. Initialized to zero when calling DllpDataSend (before DLLP has been sent).
  PCIE_DLLPSTATUS_WRONGCRC_DONE,
  ///< Returns '1' if a DLLP matched the PCIE_DLLPPROPERTY_WRONGCRCPATTERN 
  ///< and PCIE_DLLPPROPERTY_WRONGCRCMASK properties and was sent with a false LCRC.
  ///< Use DllpStatusClear() to clear the Status.
  PCIE_DLLPSTATUS_LASTSEQREQUESTED,
  ///< Returns the sequence number of last Ack/Nak sent. Read only.
  PCIE_DLLPSTATUS_LASTSEQRECEIVED
  ///< Returns the sequence number of last Ack/Nak received. Read only.
} EPCIEDllpStatus;

/// \brief Dllp control property
///
/// \use IAgtPCIEExerciser::DllpPropertySet, IAgtPCIEExerciser::DllpPropertyGet
typedef DEV_MIDL_ENUM enum EPCIEDllp
{
  PCIE_DLLPPROPERTY_WRONGCRCPATTERN,
  ///< DLLPs matching this pattern will be sent with wrong CRC. 32 bit. Default is 0.
  PCIE_DLLPPROPERTY_WRONGCRCMASK
  ///< Mask out bits for the CRC pattern match above.
  ///< Bit i=1 means that the corresponding bit in PCIE_DLLPPROPERTY_WRONGCRCPATTERN is not taken into account for match.
  ///< 32 bit. Default is 0.
} EPCIEDllp;


/// \brief Dllp control  
///
/// \use IAgtPCIEExerciser::DllpControlSet, IAgtPCIEExerciserEvents::DllpControlChanged,
///  IAgtPCIEExerciserEvents::DllpPropertyChanged
typedef DEV_MIDL_ENUM enum EPCIEDllpControl
{
  PCIE_DLLPCONTROL_WRONGCRC
  ///< Set to 1 for starting the operation. The next DLLP that matches the
  ///< PCIE_DLLPPROPERTY_WRONGCRCPATTERN and PCIE_DLLPPROPERTY_WRONGCRCMASK 
  ///< properties will be sent with corrupted LCRC. Set to 0 to cancel the operation.
} EPCIEDllpControl;

/// \brief PRP List properties
///
/// Used to set/get different PRP List properties.
typedef DEV_MIDL_ENUM enum EPCIENVMePRPListProp
{
  PCIE_NVMePRPList_DEC_INDEX,       ///< The index of the decoder to be used to map the Data Memory Internal address and Physical Memory address.
  PCIE_NVMePRPList_PAGE_NUMBER,     ///< The number of the page to be used to map the Data Memory Internal address and Physical Memory address.
  PCIE_NVMePRPList_DATA_MEM_ADDR,   ///< The Data Memory Internal address indicated by the decoder index and page number. This property is read only.
  PCIE_NVMePRPList_PHY_MEM_ADDR_LO, ///< The lower part of the Physical Memory address indicated by the decoder index and page number. This property is read only.
  PCIE_NVMePRPList_PHY_MEM_ADDR_HI, ///< The high part of the Physical Memory address indicated by the decoder index and page number. This property is read only.
  PCIE_NVMePRPList_NUM_ENTRIES,     ///< The number of entries in the PRP list. This property is read only.
} EPCIENVMePRPListProp;

/// \brief PRP Entry properties
///
/// Used to set/get different PRP Entry properties.
typedef DEV_MIDL_ENUM enum EPCIENVMePRPListEntryProp
{
  PCIE_NVMePRPListEntry_DEC_INDEX,        ///< The index of the decoder to be used to map the Data Memory Internal address and Physical Memory address.
  PCIE_NVMePRPListEntry_PAGE_NUMBER,      ///< The number of the page to be used to map the Data Memory Internal address and Physical Memory address.
  PCIE_NVMePRPListEntry_OFFSET,           ///< The value of the offset within the page. This value can only be set for entry number 0.
  PCIE_NVMePRPListEntry_DATA_MEM_ADDR,    ///< The Data Memory Internal address indicated by the decoder index and page number. This property is read only.
  PCIE_NVMePRPListEntry_PHY_MEM_ADDR_LO,  ///< The lower part of the Physical Memory address indicated by the decoder index and page number. This property is read only.
  PCIE_NVMePRPListEntry_PHY_MEM_ADDR_HI   ///< The high part of the Physical Memory address indicated by the decoder index and page number. This property is read only.
} EPCIENVMePRPListEntryProp;

/// \brief NVM Express Submission Queue packet properties
///
/// Used to set/get different NVM Express Submission Queue packet properties.
typedef DEV_MIDL_ENUM enum EPCIENVMeSQPacketProp
{
  PCIE_NVMeSQPacket_DW00,               ///< DWord 0 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW01,               ///< DWord 1 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW02,               ///< DWord 2 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW03,               ///< DWord 3 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW04,               ///< DWord 4 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW05,               ///< DWord 5 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW06,               ///< DWord 6 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW07,               ///< DWord 7 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW08,               ///< DWord 8 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW09,               ///< DWord 9 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW10,               ///< DWord 10 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW11,               ///< DWord 11 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW12,               ///< DWord 12 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW13,               ///< DWord 13 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW14,               ///< DWord 14 of the Submission Queue packet.
  PCIE_NVMeSQPacket_DW15,               ///< DWord 15 of the Submission Queue packet.
  PCIE_NVMeSQPacket_PRPList1_Id,        ///< The Id of the PRP List to be used as PRP 1 of the packet when PRP 1 is a PRP list.
  PCIE_NVMeSQPacket_PRPList2_Id,        ///< The Id of the PRP List to be used as PRP 2 of the packet when PRP 2 is a PRP list.
  PCIE_NVMeSQPacket_PRP1_DecoderIndex,  ///< The index of the decoder to be used in PRP 1 of the packet when PRP 1 is a PRP Entry.
  PCIE_NVMeSQPacket_PRP2_DecoderIndex,  ///< The index of the decoder to be used in PRP 2 of the packet when PRP 2 is a PRP Entry.
  PCIE_NVMeSQPacket_PRP1_PageNum,       ///< The number of the page to be used in PRP 1 of the packet when PRP 1 is a PRP Entry.
  PCIE_NVMeSQPacket_PRP2_PageNum,       ///< The number of the page to be used in PRP 1 of the packet when PRP 1 is a PRP Entry.
  PCIE_NVMeSQPacket_PRP1_Offset,        ///< The offset within the page to be used in PRP 1 of the packet when PRP 1 is a PRP Entry.
  PCIE_NVMeSQPacket_PRP2_IsPRPList      ///< This field is to be set if PRP 2 is a PRP list. This is used for packets where PRP 2 can be either a PRP entry or a PRP list.
} EPCIENVMeSQPacketProp;

/// \brief NVM Express Completion Queue packet properties
///
/// Used to get different NVM Express Completion Queue packet properties.
typedef DEV_MIDL_ENUM enum EPCIENVMeCQPacketProp
{
  PCIE_NVMeCQPacket_DW00,               ///< DWord 0 of the Completion Queue packet.
  PCIE_NVMeCQPacket_DW01,               ///< DWord 1 of the Completion Queue packet.
  PCIE_NVMeCQPacket_DW02,               ///< DWord 2 of the Completion Queue packet.
  PCIE_NVMeCQPacket_DW03                ///< DWord 3 of the Completion Queue packet.
} EPCIENVMeCQPacketProp;

#ifndef COMPILING_DOC
// These calls have been documented, but should not be shown to the customer 
// according to SAD00238559

/// \brief LTSSM LA Location
///
typedef DEV_MIDL_ENUM enum EPCIELTSSMLALocation
{
  PCIE_LTSSMLALOCATION_FIFOVALUE,
  ///< Used to read the corresponding value out of the current line of the FIFO of the LTSSM LA. This location is read-only.
  PCIE_LTSSMLALOCATION_TRIGGER_MASK,
  ///< Used to read or write the trigger mask of the LTSSM LA
  PCIE_LTSSMLALOCATION_TRIGGER_LEVEL,
  ///< Used to read or write the trigger level of the LTSSM LA
  PCIE_LTSSMLALOCATION_CHANGE_MASK,
  ///< Used to read or write the change mask of the LTSSM LA
} EPCIELTSSMLALocation;


/// \brief LTSSM LA  
///
typedef DEV_MIDL_ENUM enum EPCIELTSSMLA
{
  PCIE_LTSSMLA_LANE_00_RECEIVE_INFO = 0,
  ///< This is the receive info for lane 0.
  PCIE_LTSSMLA_LANE_01_RECEIVE_INFO = 1,
  ///< This is the receive info for lane 1.
  PCIE_LTSSMLA_LANE_02_RECEIVE_INFO = 2,
  ///< This is the receive info for lane 2.
  PCIE_LTSSMLA_LANE_03_RECEIVE_INFO = 3,
  ///< This is the receive info for lane 3.
  PCIE_LTSSMLA_LANE_04_RECEIVE_INFO = 4,
  ///< This is the receive info for lane 4.
  PCIE_LTSSMLA_LANE_05_RECEIVE_INFO = 5,
  ///< This is the receive info for lane 5.
  PCIE_LTSSMLA_LANE_06_RECEIVE_INFO = 6,
  ///< This is the receive info for lane 6.
  PCIE_LTSSMLA_LANE_07_RECEIVE_INFO = 7,
  ///< This is the receive info for lane 7.
  PCIE_LTSSMLA_LANE_08_RECEIVE_INFO = 8,
  ///< This is the receive info for lane 8.
  PCIE_LTSSMLA_LANE_09_RECEIVE_INFO = 9,
  ///< This is the receive info for lane 9.
  PCIE_LTSSMLA_LANE_10_RECEIVE_INFO = 10,
  ///< This is the receive info for lane 10.
  PCIE_LTSSMLA_LANE_11_RECEIVE_INFO = 11,
  ///< This is the receive info for lane 11.
  PCIE_LTSSMLA_LANE_12_RECEIVE_INFO = 12,
  ///< This is the receive info for lane 12.
  PCIE_LTSSMLA_LANE_13_RECEIVE_INFO = 13,
  ///< This is the receive info for lane 13.
  PCIE_LTSSMLA_LANE_14_RECEIVE_INFO = 14,
  ///< This is the receive info for lane 14.
  PCIE_LTSSMLA_LANE_15_RECEIVE_INFO = 15,
  ///< This is the receive info for lane 15.
  PCIE_LTSSMLA_TIMESTAMP,
  ///< This is the timestamp (32 bits). Please see the setting of PCIE_LTSSMLACONTROL_LA_TIMESTAMP_DIVIDER
  ///< in \ref EPCIELTSSMLAControl for an interpretation of this value. This property is only valid with the \ref EPCIELTSSMLAType
  ///< PCIE_LTSSMLATYPE_FIFOVALUE. This property is read-only.
  PCIE_LTSSMLA_DLCMSM_STATE,
  // (2 bit) (TBD please provide info) (Obsolete, not commented on intention)
  PCIE_LTSSMLA_MGT_DESKEW_DONE,
  ///< (1 bit) (TBD please provide info) 
  PCIE_LTSSMLA_RECEIVED_DATA_RATE,
  ///< (9 bit) (TBD please provide info) 
  PCIE_LTSSMLA_RECEIVED_NUMBER_OF_FTS,
  // (8 bit) (TBD please provide info) (Obsolete, not commented on intention)
  PCIE_LTSSMLA_NUMBER_OF_ACTIVE_LANES,
  ///< (5 bit) (TBD please provide info) 
  PCIE_LTSSMLA_LINKUP,
  ///< (1 bit) (TBD please provide info) 
  PCIE_LTSSMLA_TX_COMMAND,
  ///< (11 bit) (TBD please provide info) 
  PCIE_LTSSMLA_MGT_RX_LOSS_OF_SYNC,
  ///< (16 bit) (TBD please provide info)   
  PCIE_LTSSMLA_SW_STATE,
  ///< This is the LTSSM State as seen by the software. Please see \ref EPCIELTSSMVal for interpretation. (5bits)
  ///< When writing this value (trigger mask, trigger level or change mask) the values for PCIE_LTSSMLA_RX_L0S_STATE
  ///< will be set to 0x0.
  PCIE_LTSSMLA_FPGA_STATE,
  ///< This is the LTSSM State as seen by the FGPA. (6bits)
  PCIE_LTSSMLA_RX_L0S_STATE,
  ///< (2 bit) This is read only. Please see PCIE_LTSSMLA_SW_STATE. (TBD please provide info) 
  PCIE_LTSSMLA_PCI_WAKE,
  ///< (1 bit) (TBD please provide info) 
  PCIE_LTSSMLA_PCI_RESET,
  ///< (1 bit) (TBD please provide info) 
  PCIE_LTSSMLA_RST_N_DETECT,
  ///< (1 bit) (TBD please provide info) 
  PCIE_LTSSMLA_MGT_RX_ELECTRICAL_IDLE,
  ///< (16 bit) (TBD please provide info)
} EPCIELTSSMLA;

/// \brief LTSSM LA Control  
///
typedef DEV_MIDL_ENUM enum EPCIELTSSMLAControl
{
  PCIE_LTSSMLACONTROL_LA_ARM,
  ///< Setting to 1 clears the memory, resets the pointers and waits for trigger. Setting to 0 has
  ///< no effect. (default 0, maximum 1)
  PCIE_LTSSMLACONTROL_LA_SW_TRIGGER,
  ///< Setting to 1 executes the trigger. Setting to 0 has no effect. (default 0, maximum 1)
  PCIE_LTSSMLACONTROL_LA_REQUEST_NEXT,
  ///< Setting to 1 requests the next trigger line. You can then use the calls 
  ///< \ref Whatever to read out the LA lines. Setting to 0 has no effect. (default 0, maximum 1)
  PCIE_LTSSMLACONTROL_LA_STOP,
  ///< Setting to 1 will stop the capturing until the next arm/trigger occurs. (default 0, maximum 1)
  PCIE_LTSSMLACONTROL_LA_CENTERED_TRIGGER_POS,
  ///< Setting to 1 sets the trigger position into the middle of the LA Memory. Setting to 0 sets
  ///< the position to the beginning. (default is 0, maximum 1)
  PCIE_LTSSMLACONTROL_LA_TIMESTAMP_DIVIDER,
  ///< The timestamp is only used in transition mode. Timestamp counts every 8ns cycle
  ///< when timestamp divider is 0. Set timestamp divider to 1 to count every other 
  ///< cycle, and so on. (default 0, maximum 0xffff)
} EPCIELTSSMLAControl;


/// \brief LTSSM LA Status
///
typedef DEV_MIDL_ENUM enum EPCIELTSSMLAStatus
{
  PCIE_LTSSMLASTATUS_DONE,
  ///< If the value is 1, the memory is full. The bit gets cleared when arming the LA, see \ref TheCall
  ///< with property PCIE_LTSSMLACONTROL_LA_ARM
  PCIE_LTSSMLASTATUS_LINE_VALID,
  ///< If the value is 1, the current memory line is valid.
  PCIE_LTSSMLASTATUS_TRIGGERED,
  ///< If the value is 1, the LA triggered.
} EPCIELTSSMLAStatus;

/// \brief InSystem Port Scratch pad register number
///
typedef DEV_MIDL_ENUM enum EPCIEScratchPadRegister
{
  PCIE_SCRATCH_PAD_REGISTER0,
  ///< InSystem Port Scratch Pad Register 0
  PCIE_SCRATCH_PAD_REGISTER1,
  ///< InSystem Port Scratch Pad Register 1
  PCIE_SCRATCH_PAD_REGISTER2,
  ///< InSystem Port Scratch Pad Register 2
  PCIE_SCRATCH_PAD_REGISTER3,
  ///< InSystem Port Scratch Pad Register 3
} EPCIEScratchPadRegister;


#endif    // #ifndef COMPILING_DOC


/// @} end of group Exerciser in this file

#ifndef COMPILING_DOC
/// \addtogroup Analyzer
///
/// @{

/// \brief Scramble enable/disable/auto-detect
///
typedef DEV_MIDL_ENUM enum EPCIEScramble
{
  PCIE_SCRAMBLE_DISABLE = 0,  ///< Scrambling forced off
  PCIE_SCRAMBLE_ENABLE = 1, ///< Scrambling forced on
  PCIE_SCRAMBLE_AUTOMATIC = 2 ///< Scrambling auto detect (during link training)
} EPCIEScramble;

/// \brief Analyzer properties
///
typedef DEV_MIDL_ENUM enum EPCIEAnalyzerProp
{
  PCIE_MAX_TRACEMEMORY_DEPTH, ///< The maximum trace memory depth (read only)
  PCIE_TRACEMEMORY_DEPTH,   ///< The currently used trace memory depth
  PCIE_LINKWIDTH,       ///< Link width (see \ref EPCIELinkWidthVal) (bits 0-7 = upstream, bits 8-15 = downstream)
  PCIE_SCRAMBLE,        ///< Scrambling enable/disable/automatic (bits 0-7 = upstream, bits 8-15 = downstream)
  PCIE_TRIGGER_POSITION,    ///< Trigger position (0...100)
  PCIE_LANE_SELECT,     ///< Selects on which physical lane a link is expected (not implemented yet)
  PCIE_RX_RESET,        ///< Resets the MGT receiver (bit 0=upstream, bit8=downstream)
  PCIE_LANE_REVERSAL,     ///< Enables lane reversal (bit 0=upstream, bit8=downstream)
  PCIE_AUTO_POLARITY_DETECT,  ///< Enable automatic polarity detection (bit 0=upstream, bit8=downstream)
  PCIE_POLARITY,        ///< Select inverted polarity (bits 0-7=upstream, bits 8-15=downstream)
  PCIE_SPEC_REVISION,     ///< Select specification revision (0=Rev 1.0, 1=Rev 1.0a). Default is 1.0a
  PCIE_CLOCK_SOURCE,      ///< Selects the clock source for the analyzer (0=internal 1=external)
  PCIE_OUTPUT_ENABLE_DELAY,   // When the PCI Reset is deasserted, wait this period of time until the outputs are enabled. Unit is milli seconds. (default 250).
  PCIE_DUT_CONNECTIVITY,      ///< Enables handling of pci resets. Please refer to \ref EPCIEDUTConnectivityVal. Default is 1 (PCIE_DUT_CONNECTIVITYV_PLATFORM).
  PCIE_FLIP_DIRECTION     ///< Flip upstream & downstream. This will be automatically set based on the probe board.
} EPCIEAnalyzerProp;

typedef DEV_MIDL_ENUM enum EPCIEAnalyzerStatus
{
  PCIE_STAT_IS_RUNNING,   ///< 1 if analyzer is running, 0 if stopped
  PCIE_STAT_HAS_TRIGGERED,  ///< 1 if analyzer has triggered
  PCIE_STAT_PERCENT_FULL,   ///< 0...100 (%) memory full
  PCIE_STAT_LINKWIDTH,    ///< detected link width (bits 0-7=upstream, bits 8-15=downstream). A value of 0 means no linkwidth detected yet
  PCIE_STAT_SCRAMBLE,     ///< detected scrambling (bit 0=upstream, bit 8=downstream)
  PCIE_STAT_LANE_REVERSAL,  ///< detected lane reversal (bit 0=upstream, bit8=downstream)
  PCIE_STAT_POLARITY,     ///< detected inverted polarity (bits 0-7=upstream, bits 8-15=downstream). Bits 16..23 and 24..31 indicate if the polarity status is valid(0) or unknown(1)
  PCIE_STAT_COUNTER1,     ///< reports the current value of counter1
  PCIE_STAT_COUNTER2      ///< reports the current value of counter2
} EPCIEAnalyzerStatus;

// There is a copy of this enum in the exerciser ! TBD move stuff in control interface
/// \brief PCI Express DUT Connectivity values
///
typedef DEV_MIDL_ENUM enum EPCIEDUTConnectivityVal
{
  PCIE_DUT_CONNECTIVITYV_BENCH = 0,   ///< (0) The probeboard is not inserted into a platform.
  PCIE_DUT_CONNECTIVITYV_PLATFORM = 1 ///< (1) The probeboard is inserted into a plaform.
} EPCIEDUTConnectivityVal;

/// @} end of group Analyzer in this file
#endif
// -----------------------------------------------------------------------

#endif /* DEVPCIEDEFS_H */